package ru.yandex.partner.core.entity.agreement.service;

import java.time.LocalDate;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.annotations.VisibleForTesting;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.block.BlockType;
import ru.yandex.partner.core.entity.ModelQueryService;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.user.filter.UserFilters;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.service.integration.balance.BalanceDocumentService;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.BalancePartnerContract;
import ru.yandex.partner.libs.extservice.balance.method.partnercontract.Contract;

import static com.google.common.collect.Maps.newHashMapWithExpectedSize;
import static ru.yandex.partner.core.filter.CoreFilterNode.eq;

@Service
public class AgreementChecker {
    private final ProductRunClearance productRunClearance;
    private final BalanceDocumentService balanceDocumentService;
    private final ModelQueryService<User> userQueryService;
    private final AgreementCheckerProperties props;

    public AgreementChecker(ProductRunClearance productRunClearance,
                            BalanceDocumentService balanceDocumentService,
                            ModelQueryService<User> userQueryService,
                            AgreementCheckerProperties props) {
        this.productRunClearance = productRunClearance;
        this.balanceDocumentService = balanceDocumentService;
        this.userQueryService = userQueryService;
        this.props = props;
    }

    public boolean hasAgreementForAnyProductForToday(long clientId, Collection<BlockType> products) {
        return hasAgreementForAnyProductForToday(clientId, products, false);
    }

    public boolean hasAgreementForAnyProductForTodayByOwner(User owner, Collection<BlockType> products) {
        return hasAgreementForAnyProductForDay(owner.getClientId(), owner, products, LocalDate.now(), false);
    }

    public boolean hasAgreementForAnyProductForToday(long clientId, Collection<BlockType> products,
                                                     boolean notCheckAssessor) {
        return hasAgreementForAnyProductForDay(clientId, products, LocalDate.now(), notCheckAssessor);
    }

    public boolean hasAgreementForAnyProductForTomorrow(long clientId, Collection<BlockType> products) {
        return hasAgreementForAnyProductForTomorrow(clientId, products, false);
    }

    public boolean hasAgreementForAnyProductForTomorrow(long clientId, Collection<BlockType> products,
                                                        boolean notCheckAssessor) {
        return hasAgreementForAnyProductForDay(clientId, products, LocalDate.now().plusDays(1), notCheckAssessor);
    }

    private boolean hasAgreementForAnyProductForDay(long clientId, Collection<BlockType> products, LocalDate day,
                                                    boolean notCheckAssessor) {
        var user = findUserByClientId(clientId);
        return hasAgreementForAnyProductForDay(clientId, user, products, day, notCheckAssessor);
    }

    private boolean hasAgreementForAnyProductForDay(long clientId, User user, Collection<BlockType> products,
                                                    LocalDate day,
                                                    boolean notCheckAssessor) {
        if (!notCheckAssessor) {
            if (user != null && Boolean.TRUE.equals(user.getIsAssessor())) {
                return true;
            }
        }

        return products.stream().anyMatch(
                getDataForClient(clientId, user, day).get(day)::contains
        );
    }


    private @Nullable
    User findUserByClientId(long clientId) {
        List<User> users = userQueryService.findAll(
                QueryOpts.forClass(User.class)
                        .withFilter(eq(UserFilters.CLIENT_ID, clientId))
                        .withProps(Set.of(
                                User.ID, User.CLIENT_ID, User.IS_TUTBY,
                                User.HAS_TUTBY_AGREEMENT, User.IS_ASSESSOR
                                )
                        )
        );
        return users.isEmpty() ? null : users.get(0);
    }

    private User findUserId(long userId) {
        return userQueryService.findAll(
                QueryOpts.forClass(User.class)
                        .withFilter(eq(UserFilters.ID, userId))
                        .withProps(Set.of(
                                User.ID,
                                User.CLIENT_ID,
                                User.IS_TUTBY,
                                User.HAS_TUTBY_AGREEMENT,
                                User.IS_ASSESSOR))
                ).get(0);
    }


    /**
     * Собирает информацию о том, какие продукты могут работать в заданые дни
     *
     * @param clientId id клиента
     * @param user     user в нашей системе (допускается остутствие)
     * @param days     дни, в которые проверяются продукты
     * @return маппинг дата -> список продуктов, которые могут работать в эту дату
     */
    private Map<LocalDate, Set<BlockType>> getDataForClient(Long clientId, User user, LocalDate... days) {
        if (isSpecialClientIdWithAllProducts(clientId)) {
            return forAllDays(EnumSet.allOf(BlockType.class), days);
        }

        var vipProducts = props.getVips().get(clientId);
        if (vipProducts != null) {
            return forAllDays(vipProducts, days);
        }

        if (user != null && Boolean.TRUE.equals(user.getIsTutby())
                && Boolean.TRUE.equals(user.getHasTutbyAgreement())) {
            return forAllDays(props.getTutbyProducts(), days);
        }

        // честно считаем допустимые продукты по дням и действующим в эти дни договорам
        List<BalancePartnerContract> contracts = balanceDocumentService.getPartnerContracts(clientId);
        Map<LocalDate, Set<BlockType>> allowedTypesPerDay = newHashMapWithExpectedSize(days.length);

        for (LocalDate day : days) {
            EnumSet<BlockType> allowedTypes = EnumSet.noneOf(BlockType.class);
            allowedTypesPerDay.put(day, allowedTypes);
            for (BalancePartnerContract contract : contracts) {
                for (BlockType product : BlockType.values()) {
                    if (!allowedTypes.contains(product) && canRun(contract, product, day)) {
                        allowedTypes.add(product);
                    }
                }
            }
        }

        return allowedTypesPerDay;
    }

    private Map<LocalDate, Set<BlockType>> forAllDays(Set<BlockType> blockTypes, LocalDate... days) {
        return Stream.of(days).collect(Collectors.toMap(
                Function.identity(),
                x -> blockTypes
        ));
    }

    /**
     * Можно ли по договору (если он действует на заданный день) продолжать работу продукта
     *
     * @param balanceContract договор клиента
     * @param productType     тип продукта
     * @param day             день
     * @return true, если работа продукта допустима
     */
    private boolean canRun(
            BalancePartnerContract balanceContract,
            BlockType productType,
            LocalDate day
    ) {
        if (!balanceContract.isLive(day)) {
            return false;
        }

        Optional<Contract> contractOpt = balanceContract.getContract();

        if (contractOpt.isEmpty()) {
            return false;
        }

        var contract = contractOpt.get();
        var contractTypeId = contract.getContractType();

        var canRunIfSignedOrFaxed =
                productRunClearance.canRunWithSignType(productType, contractTypeId, SignType.SIGNED_OR_FAXED);

        return canRunIfSignedOrFaxed && (contract.isSignedOrFaxed() || contract.isOferta2TestMode());
    }

    public boolean isSpecialClientIdWithAllProducts(long clientId) {
        return props.getSpecialClients().contains(clientId);
    }

    @VisibleForTesting
    Map<LocalDate, Set<BlockType>> getDataForClient(long clientId, LocalDate... days) {
        return getDataForClient(
                clientId,
                findUserByClientId(clientId),
                days
        );
    }
}
