package ru.yandex.partner.core.entity.agreement.service;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.PostConstruct;

import com.google.common.collect.Sets;
import com.univocity.parsers.common.Context;
import com.univocity.parsers.common.processor.core.Processor;
import com.univocity.parsers.csv.CsvFormat;
import com.univocity.parsers.csv.CsvParser;
import com.univocity.parsers.csv.CsvParserSettings;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.block.BlockType;

@Component
class ParsedProductRunClearance implements Processor<Context> {
    private final Resource mappingFile;
    private final CsvParserSettings parserSettings;

    private Map<Key, SignType> allowedKeys;
    private Set<BlockType> knownProductTypes;
    private Set<Integer> knownContractTypes;

    ParsedProductRunClearance(
            @Value("classpath:allow_run_products_for_contract_types.tsv") Resource mappingFile) {
        this.mappingFile = mappingFile;
        this.parserSettings = new CsvParserSettings();
        this.parserSettings.setHeaderExtractionEnabled(true);
        CsvFormat tsvFormat = new CsvFormat();
        tsvFormat.setLineSeparator("\n");
        tsvFormat.setDelimiter('|');
        this.parserSettings.setFormat(tsvFormat);
    }

    @PostConstruct
    public void parseConfig() {
        parserSettings.setProcessor(this);
        var tsvParser = new CsvParser(parserSettings);
        try (var is = mappingFile.getInputStream()) {
            tsvParser.parse(is);
        } catch (IOException e) {
            throw new RuntimeException("Could not read resource: " + mappingFile, e);
        }

        Set<BlockType> blockTypesWithContract = Stream.of(BlockType.values())
                .filter(BlockType::isHasContract)
                .collect(Collectors.toSet());
        Sets.SetView<BlockType> nonConfiguredProducts =
                Sets.difference(blockTypesWithContract, knownProductTypes);
        if (!nonConfiguredProducts.isEmpty()) {
            throw new IllegalStateException("Не настроены продукты: " + nonConfiguredProducts);
        }
    }

    @Override
    public void processStarted(Context context) {
        this.allowedKeys = new HashMap<>();
        this.knownProductTypes = new HashSet<>();
    }

    @Override
    public void rowProcessed(String[] row, Context context) {
        BlockType productType = BlockType.from(row[0]);
        knownProductTypes.add(productType);
        for (int i = 1; i < row.length; i++) {
            if (row[i] == null) {
                continue;
            }
            int contractType = Integer.parseInt(context.headers()[i]);
            allowedKeys.put(new Key(productType, contractType), SignType.from(row[i]));
        }
    }

    @Override
    public void processEnded(Context context) {
        this.knownContractTypes = Stream.of(context.headers()).skip(1)
                .filter(Objects::nonNull)
                .map(Integer::parseInt)
                .collect(Collectors.toUnmodifiableSet());
        this.knownProductTypes = Collections.unmodifiableSet(knownProductTypes);
        this.allowedKeys = Collections.unmodifiableMap(this.allowedKeys);
    }

    public Map<Key, SignType> getAllowedKeys() {
        return allowedKeys;
    }

    public Set<BlockType> getKnownProductTypes() {
        return knownProductTypes;
    }

    public Set<Integer> getKnownContractTypes() {
        return knownContractTypes;
    }

    record Key(BlockType productType, Integer contractType) {
    }
}
