package ru.yandex.partner.core.entity.block.actions;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.partner.core.action.AbstractActionContextWithModelProperty;
import ru.yandex.partner.core.action.ActionErrorHandler;
import ru.yandex.partner.core.action.ActionModelContainerImpl;
import ru.yandex.partner.core.action.log.ActionsLogger;
import ru.yandex.partner.core.entity.IncomingFields;
import ru.yandex.partner.core.entity.ModelCopyMapper;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.service.BlockService;
import ru.yandex.partner.core.entity.block.service.BlockUpdateOperationFactory;
import ru.yandex.partner.core.filter.CoreFilterNode;

public abstract class BlockActionContext<B extends BaseBlock> extends AbstractActionContextWithModelProperty<B> {
    private final BlockService blockService;
    private final BlockUpdateOperationFactory blockUpdateOperationFactory;
    private final ModelCopyMapper<B> blockCopyMapper;

    public BlockActionContext(ActionsLogger actionsLogger, ActionErrorHandler<B> errorHandler,
                              BlockService blockService, ModelCopyMapper<B> blockCopyMapper,
                              BlockUpdateOperationFactory blockUpdateOperationFactory) {
        super(actionsLogger, errorHandler);
        this.blockService = blockService;
        this.blockCopyMapper = blockCopyMapper;
        this.blockUpdateOperationFactory = blockUpdateOperationFactory;
    }

    @Override
    protected List<B> getModelsByIds(
            Collection<Long> ids, Set<ModelProperty<? extends Model, ?>> fields,
            CoreFilterNode<B> filter, boolean forUpdate) {
        return blockService.findAll(
                QueryOpts.forClass(getEntityClass())
                        .withFilterByIds(ids)
                        .withFilter(filter)
                        .withProps(fields)
                        .forUpdate(forUpdate)
        );
    }

    @Override
    public Operation<Long> getUpdateOperation(
            List<ModelChanges<B>> modelChanges,
            IncomingFields incomingFields,
            boolean rollbackIfErrors
    ) {
        List<ActionModelContainerImpl<B>> containersWithoutErrors =
                this.getContainersWithoutErrors(modelChanges.stream().map(ModelChanges::getId)
                        .collect(Collectors.toList()));

        return blockUpdateOperationFactory.createUpdateOperationWithPreloadedModels(
                rollbackIfErrors ? Applicability.FULL : Applicability.PARTIAL,
                modelChanges.stream().map(
                        it -> it.castModel(BaseBlock.class)
                ).collect(Collectors.toList()),
                incomingFields,
                containersWithoutErrors.stream().map(ActionModelContainerImpl::getNonChangedItem)
                        .collect(Collectors.toList()),
                getModelStubCreator()
        );
    }

    @Override
    protected B clone(B item) {
        return blockCopyMapper.copy(item);
    }

    @Override
    protected ModelChanges<B> createModelChanges(Long id) {
        return new ModelChanges<>(id, getEntityClass());
    }

    protected abstract Function<Long, BaseBlock> getModelStubCreator();
}
