package ru.yandex.partner.core.entity.block.actions.all;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import one.util.streamex.StreamEx;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.action.ActionConfiguration;
import ru.yandex.partner.core.action.ActionContext;
import ru.yandex.partner.core.action.ActionContextFacade;
import ru.yandex.partner.core.action.ActionContextWithModelProperty;
import ru.yandex.partner.core.action.ActionErrorHandler;
import ru.yandex.partner.core.action.ActionModelContainer;
import ru.yandex.partner.core.action.ActionModelContainerImpl;
import ru.yandex.partner.core.action.ActionPerformer;
import ru.yandex.partner.core.action.factories.block.ActionSetNeedUpdateFactory;
import ru.yandex.partner.core.action.factories.block.ActionStartFactory;
import ru.yandex.partner.core.action.helper.ActionOptsHelper;
import ru.yandex.partner.core.entity.block.actions.BlockAction;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithMultistate;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.entity.page.model.PageWithMultistate;
import ru.yandex.partner.core.multistate.block.BlockStateFlag;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.core.service.entitymanager.EntityManager;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;

public class BlockActionAdd<B extends BlockWithMultistate> extends BlockAction<B> {
    private final Map<Long, B> addedModels;
    private final Map<Long, Set<ModelProperty<? super B, ?>>> addedProperties;
    private final ObjectMapper objectMapper;
    private final ActionSetNeedUpdateFactory<B, BlockStateFlag> actionSetNeedUpdateFactory;
    private final ActionStartFactory<B, BlockStateFlag> blockStartFactory;
    private final ActionContextFacade actionContextFacade;
    private final EntityManager entityManager;
    private final BiConsumer<Map<Long, Long>, ActionContextFacade> blocksCountIncrementor;

    @SuppressWarnings("ParameterNumber")
    public BlockActionAdd(
            ActionConfiguration<B, ?> parentFactory,
            String name,
            Map<Long, B> addedModels,
            Map<Long, Set<ModelProperty<? super B, ?>>> addedProperties,
            MultistateGraph<B, BlockStateFlag> multistateGraph,
            ActionPerformer actionPerformer,
            ActionErrorHandler<B> actionErrorHandler,
            ObjectMapper objectMapper,
            Class<B> blockClass,
            ActionSetNeedUpdateFactory<B, BlockStateFlag> actionSetNeedUpdateFactory,
            ActionStartFactory<B, BlockStateFlag> blockStartFactory, EntityManager entityManager,
            BiConsumer<Map<Long, Long>, ActionContextFacade> blocksCountIncrementor) {
        super(
                parentFactory,
                name,
                addedModels.keySet(),
                multistateGraph,
                actionPerformer,
                actionErrorHandler,
                blockClass
        );
        this.addedModels = addedModels;
        this.addedProperties = addedProperties;
        this.objectMapper = objectMapper;
        this.actionSetNeedUpdateFactory = actionSetNeedUpdateFactory;
        this.blockStartFactory = blockStartFactory;
        this.actionContextFacade = actionPerformer.getActionContextFacade();
        this.entityManager = entityManager;
        this.blocksCountIncrementor = blocksCountIncrementor;
    }

    @Override
    public String getSerializedOpts(Long id) {
        B addedModel = addedModels.get(id);
        Set<ModelProperty<? super B, ?>> changedProps = addedProperties.get(id);
        return ActionOptsHelper.getSerializedOptsForAdd(id,
                objectMapper, getName(), getEntityClass(), addedModel, changedProps);
    }

    @Override
    public void onAction(ActionContext<B, ActionModelContainerImpl<B>> bActionModelContainerActionContext,
                         List<ActionModelContainerImpl<B>> containers) {
        super.onAction(bActionModelContainerActionContext, containers);

        var pageClazz = entityManager.getPageClassByBlockClass(blockClass, BasePage.class);

        var pageIds =
                StreamEx.of(containers).map(ActionModelContainerImpl::getItem).map(BaseBlock::getPageId)
                        .collect(Collectors.groupingBy(Function.identity(), Collectors.counting()));

        var pageContainers = actionContextFacade.getActionContext(pageClazz)
                .getContainers(
                        pageIds.keySet(),
                        Set.of(PageWithMultistate.MULTISTATE, BasePage.ID),
                        true
                );

        var pageContainersMap = StreamEx.of(pageContainers)
                .collect(Collectors.toMap(c -> c.getItem().getId(), ActionModelContainer::getItem));

        List<Long> blockIds = Lists.newArrayListWithExpectedSize(containers.size());
        // we can only start block on a page in testing/working multistate
        List<Long> blockIdsToStart = Lists.newArrayListWithExpectedSize(containers.size());

        for (var container : containers) {
            blockIds.add(container.getItem().getId());

            var pageMultistate = ((PageWithMultistate) pageContainersMap.get(container.getItem().getPageId()))
                    .getMultistate();

            if (pageMultistate.test(PageStateFlag.WORKING.or(PageStateFlag.TESTING))) {
                blockIdsToStart.add(container.getItem().getId());
            }
        }

        var setNeedUpdateAction = actionSetNeedUpdateFactory.createAction(blockIds);
        var startAction = blockStartFactory.createAction(blockIdsToStart);

        doNestedActions(startAction, setNeedUpdateAction);

        blocksCountIncrementor.accept(pageIds, actionContextFacade);
    }

    @Override
    public void prepareContext(ActionContextWithModelProperty<B, ?> actionContext) {
        super.prepareContext(actionContext);
        for (Map.Entry<Long, B> entry : addedModels.entrySet()) {
            actionContext.preloadModels(
                    addedProperties.get(entry.getKey())
                            .stream()
                            .map(modelProperty -> (ModelProperty<? extends Model, ?>) modelProperty)
                            .collect(Collectors.toUnmodifiableSet()),
                    List.of(entry.getValue()));
        }
    }
}
