package ru.yandex.partner.core.entity.block.actions.all;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.action.ActionConfiguration;
import ru.yandex.partner.core.action.ActionContext;
import ru.yandex.partner.core.action.ActionContextWithModelProperty;
import ru.yandex.partner.core.action.ActionErrorHandler;
import ru.yandex.partner.core.action.ActionModelContainerImpl;
import ru.yandex.partner.core.action.ActionPerformer;
import ru.yandex.partner.core.action.ActionUtils;
import ru.yandex.partner.core.action.WithUpdatedFields;
import ru.yandex.partner.core.action.exception.ActionError;
import ru.yandex.partner.core.action.factories.block.ActionSetNeedUpdateFactory;
import ru.yandex.partner.core.action.helper.ActionOptsHelper;
import ru.yandex.partner.core.entity.IncomingFields;
import ru.yandex.partner.core.entity.block.actions.BlockAction;
import ru.yandex.partner.core.entity.block.actions.defects.presentation.BlockActionDefectMsg;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithMultistate;
import ru.yandex.partner.core.entity.block.service.type.update.BlockUpdateOperationTypeSupportFacade;
import ru.yandex.partner.core.entity.block.service.validation.type.BlockValidationTypeSupportFacade;
import ru.yandex.partner.core.multistate.block.BlockStateFlag;
import ru.yandex.partner.core.validation.defects.DefectInfoBuilder;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;


/**
 * Экшен для редактирования блоков
 */
@ParametersAreNonnullByDefault
public class BlockActionEdit<B extends BlockWithMultistate> extends BlockAction<B>
        implements WithUpdatedFields {
    private final Map<Long, ModelChanges<B>> modelChangesMap;
    private final ObjectMapper objectMapper;
    private final IncomingFields incomingFields;
    private final ActionSetNeedUpdateFactory<B, BlockStateFlag> setNeedUpdateFactory;
    private final BlockUpdateOperationTypeSupportFacade blockUpdateOperationTypeSupportFacade;
    private final BlockValidationTypeSupportFacade blockValidationTypeSupportFacade;

    @SuppressWarnings("checkstyle:parameternumber")
    public BlockActionEdit(ActionConfiguration<B, ?> parentFactory,
                           String name,
                           Collection<ModelChanges<B>> modelChanges,
                           IncomingFields incomingFields,
                           MultistateGraph<B, BlockStateFlag> multistateGraph,
                           ActionPerformer actionPerformer,
                           ActionErrorHandler<B> actionErrorHandler,
                           ObjectMapper objectMapper,
                           ActionSetNeedUpdateFactory<B, BlockStateFlag> setNeedUpdateFactory,
                           BlockUpdateOperationTypeSupportFacade blockUpdateOperationTypeSupportFacade,
                           Class<B> blockClass, BlockValidationTypeSupportFacade blockValidationTypeSupportFacade) {
        this(parentFactory, name,
                modelChanges.stream().collect(Collectors.toMap(ModelChanges::getId, Function.identity())),
                incomingFields, multistateGraph, actionPerformer, actionErrorHandler,
                objectMapper, setNeedUpdateFactory, blockUpdateOperationTypeSupportFacade, blockClass,
                blockValidationTypeSupportFacade);
    }

    @SuppressWarnings("checkstyle:parameternumber")
    public BlockActionEdit(ActionConfiguration<B, ?> parentFactory,
                           String name,
                           Map<Long, ModelChanges<B>> modelChangesMap,
                           IncomingFields incomingFields,
                           MultistateGraph<B, BlockStateFlag> multistateGraph,
                           ActionPerformer actionPerformer,
                           ActionErrorHandler<B> actionErrorHandler,
                           ObjectMapper objectMapper,
                           ActionSetNeedUpdateFactory<B, BlockStateFlag> setNeedUpdateFactory,
                           BlockUpdateOperationTypeSupportFacade blockUpdateOperationTypeSupportFacade,
                           Class<B> blockClass, BlockValidationTypeSupportFacade blockValidationTypeSupportFacade) {
        super(parentFactory,
                name,
                modelChangesMap.keySet(),
                multistateGraph,
                actionPerformer,
                actionErrorHandler,
                blockClass,
                true
        );
        this.incomingFields = incomingFields;
        this.objectMapper = objectMapper;
        this.modelChangesMap = modelChangesMap;
        this.setNeedUpdateFactory = setNeedUpdateFactory;
        this.blockUpdateOperationTypeSupportFacade = blockUpdateOperationTypeSupportFacade;
        this.blockValidationTypeSupportFacade = blockValidationTypeSupportFacade;
    }


    @Override
    public Set<ModelProperty<?, ?>> getDependsOn() {
        var stream1 = super.getDependsOn().stream();
        var stream2 = modelChangesMap.values().stream().map(ModelChanges::getChangedPropsNames).flatMap(Set::stream);
        return Stream.concat(stream1, stream2).collect(Collectors.toSet());
    }

    @Override
    public void onAction(ActionContext<B, ActionModelContainerImpl<B>> context,
                         List<ActionModelContainerImpl<B>> containers) {
        for (ActionModelContainerImpl<B> container : containers) {
            ModelChanges<B> modelChanges = this.getModelChangesById(container.getProperty(BaseBlock.ID));

            modelChanges.getChangedPropsNames().forEach(modelProperty -> container.changeProperty(
                    (ModelProperty<B, Object>) modelProperty,
                    modelChanges.getChangedProp(modelProperty)
            ));
        }

        var needBsResyncIds = blockUpdateOperationTypeSupportFacade
                .getNeedBsResyncIdsByModelChanges(modelChangesMap.values());

        if (needBsResyncIds.isEmpty()) {
            return;
        }

        var setNeedUpdateAction =
                setNeedUpdateFactory.createAction(needBsResyncIds);
        doNestedActions(setNeedUpdateAction);
    }

    @Override
    public String getSerializedOpts(Long id) {
        return ActionOptsHelper.prepareSerializedOptsForUpdated(modelChangesMap.get(id), objectMapper, blockClass);
    }

    @Override
    protected void checkByAction(List<ActionModelContainerImpl<B>> actionModelContainers) {
        var errorHandler = getActionErrorHandler();
        var deleted = ActionUtils
                .filteredItemIdsByMultistateFlag(actionModelContainers, BlockStateFlag.DELETED);

        deleted.forEach(id -> errorHandler.addErrorById(id,
                DefectInfoBuilder.createDefectInfo(BlockActionDefectMsg.ARCHIVED_BLOCK),
                ActionError.ActionDefectType.CAN_NOT_DO_ACTION));
    }

    public ModelChanges<B> getModelChangesById(Long id) {
        return Objects.requireNonNull(modelChangesMap.get(id),
                "Edit action: No changes found for model: " + id);
    }

    @Override
    public IncomingFields getIncomingFields() {
        return incomingFields;
    }

    @Override
    public void prepareContext(ActionContextWithModelProperty<B, ?> actionContext) {
        Set<ModelProperty<?, ?>> changedProps = modelChangesMap.values().stream()
                .map(ModelChanges::getChangedPropsNames)
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());

        actionContext.addFieldsToRequired(
                blockValidationTypeSupportFacade.getPropertiesForValidate(
                        changedProps,
                        getEntityClass()
                )
        );
        Set<ModelProperty<?, ?>> updateAffectedProps = blockUpdateOperationTypeSupportFacade
                .getAffectedProps(modelChangesMap.values(), getEntityClass());
        actionContext.addFieldsToRequired(updateAffectedProps);

        super.prepareContext(actionContext);
    }
}
