package ru.yandex.partner.core.entity.block.actions.all;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.partner.core.action.ActionConfiguration;
import ru.yandex.partner.core.action.ActionContext;
import ru.yandex.partner.core.action.ActionErrorHandler;
import ru.yandex.partner.core.action.ActionModelContainerImpl;
import ru.yandex.partner.core.action.ActionPerformer;
import ru.yandex.partner.core.action.ActionUserIdContext;
import ru.yandex.partner.core.action.exception.ActionError;
import ru.yandex.partner.core.action.factories.block.ActionSetNeedUpdateFactory;
import ru.yandex.partner.core.action.factories.block.ActionStartFactory;
import ru.yandex.partner.core.block.BlockType;
import ru.yandex.partner.core.entity.agreement.service.AgreementChecker;
import ru.yandex.partner.core.entity.block.actions.BlockAction;
import ru.yandex.partner.core.entity.block.actions.BlockActionsEnum;
import ru.yandex.partner.core.entity.block.actions.defects.presentation.BlockActionDefectMsg;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithMultistate;
import ru.yandex.partner.core.entity.page.model.PageWithMultistate;
import ru.yandex.partner.core.entity.page.model.prop.PageWithOwnerOwnerPropHolder;
import ru.yandex.partner.core.multistate.block.BlockStateFlag;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.core.role.Role;
import ru.yandex.partner.core.validation.defects.DefectInfoBuilder;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;
import ru.yandex.partner.libs.rbac.userrole.UserRoleService;

@ParametersAreNonnullByDefault
public abstract class BlockActionRestore<B extends BlockWithMultistate,
        P extends PageWithMultistate & PageWithOwnerOwnerPropHolder>
        extends BlockAction<B> {
    private final AgreementChecker agreementChecker;
    private final ActionSetNeedUpdateFactory<B, BlockStateFlag> actionSetNeedUpdateFactory;
    private final ActionStartFactory<B, BlockStateFlag> actionStartFactory;
    private final UserRoleService userRoleService;
    private final ActionUserIdContext actionUserIdContext;

    @SuppressWarnings("checkstyle:parameternumber")
    public BlockActionRestore(ActionConfiguration<B, ?> parentFactory,
                              Collection<Long> containerIds,
                              String actionName,
                              ActionSetNeedUpdateFactory<B, BlockStateFlag> actionSetNeedUpdateFactory,
                              ActionStartFactory<B, BlockStateFlag> actionStartFactory,
                              MultistateGraph<B, BlockStateFlag> multistateGraph,
                              ActionPerformer actionPerformer,
                              ActionErrorHandler<B> actionErrorHandler,
                              Class<B> blockClass,
                              AgreementChecker agreementChecker,
                              UserRoleService userRoleService,
                              ActionUserIdContext actionUserIdContext) {
        super(parentFactory, actionName, containerIds, multistateGraph,
                actionPerformer, actionErrorHandler, blockClass);
        this.actionSetNeedUpdateFactory = actionSetNeedUpdateFactory;
        this.actionStartFactory = actionStartFactory;
        this.agreementChecker = agreementChecker;
        this.userRoleService = userRoleService;
        this.actionUserIdContext = actionUserIdContext;
    }

    @Override
    public void onAction(ActionContext<B, ActionModelContainerImpl<B>> context,
                         List<ActionModelContainerImpl<B>> containers) {
        var blocks = StreamEx.of(containers).map(ActionModelContainerImpl::getItem).toSet();
        var pages = getPages(blocks);

        List<B> validBlocks = new ArrayList<>(blocks.size());
        for (B block : blocks) {
            var curPage = pages.get(block.getPageId());
            if (curPage.getMultistate().test(PageStateFlag.WORKING)) {
                var canStart = agreementChecker.hasAgreementForAnyProductForTodayByOwner(curPage.getOwner(),
                        getBlockTypesForChecker());
                if (!canStart) {
                    getActionErrorHandler().addErrorById(block.getId(), getAgreementDefectInfo(),
                            ActionError.ActionDefectType.CAN_NOT_DO_ACTION);
                    continue;
                }
            }
            validBlocks.add(block);
        }

        List<B> allowedStartBlocks = filteredItemsByActionAllowed(BlockActionsEnum.START, validBlocks);
        List<Long> allowedStartBlockIds = StreamEx.of(allowedStartBlocks).map(BaseBlock::getId).toList();

        var start = actionStartFactory.createAction(allowedStartBlockIds);
        var setNeedUpdate =
                actionSetNeedUpdateFactory.createAction(StreamEx.of(validBlocks).map(BaseBlock::getId).toList());
        doNestedActions(start, setNeedUpdate);
        var pagesCount = StreamEx.of(validBlocks).map(BaseBlock::getPageId)
                .collect(Collectors.groupingBy(Function.identity(), Collectors.counting()));
        incrementPagesCount(pagesCount);
    }

    @Override
    public String getSerializedOpts(Long id) {
        return "{}";
    }

    protected abstract List<BlockType> getBlockTypesForChecker();

    protected abstract void incrementPagesCount(Map<Long, Long> pageIds);

    protected abstract Map<Long, P> getPages(Set<B> blocks);

    private DefectInfo<Defect> getAgreementDefectInfo() {
        //TODO: сделать это в jsonApi  PI-24889
        long userId = actionUserIdContext.get();
        if (userId == 0L) {
            return DefectInfoBuilder.createDefectInfo(BlockActionDefectMsg.INVALID_AGREEMENT);
        }
        var allRoles = userRoleService.getUserRoles(userId);
        var isInternalUser = StreamEx.of(allRoles).anyMatch(Role::isInternal);

        return isInternalUser ? DefectInfoBuilder.createDefectInfo(BlockActionDefectMsg.INVALID_AGREEMENT) :
                DefectInfoBuilder.createDefectInfo(BlockActionDefectMsg.CAN_NOT_START_WITH_YOUR_AGREEMENT);
    }
}
