package ru.yandex.partner.core.entity.block.bs

import NPartner.Page
import org.springframework.stereotype.Component
import ru.yandex.direct.dbutil.sharding.ShardSupport
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.multitype.repository.TypedRepository
import ru.yandex.partner.core.action.ActionPerformer
import ru.yandex.partner.core.bs.BkDataException
import ru.yandex.partner.core.bs.BkDataFacade
import ru.yandex.partner.core.bs.BkDataRepository
import ru.yandex.partner.core.entity.block.actions.BlockAction
import ru.yandex.partner.core.entity.block.actions.BlockActionType
import ru.yandex.partner.core.entity.block.actions.BlockActionsEnum
import ru.yandex.partner.core.entity.block.actions.all.factories.BlockMultistateActionFactory
import ru.yandex.partner.core.entity.block.container.BlockBkDictContainer
import ru.yandex.partner.core.entity.block.container.BlockContainerImpl
import ru.yandex.partner.core.entity.block.model.BaseBlock
import ru.yandex.partner.core.entity.block.model.BlockWithCustomBkData
import ru.yandex.partner.core.entity.block.model.BlockWithMultistate
import ru.yandex.partner.core.entity.block.repository.BlockModifyRepository
import ru.yandex.partner.core.entity.block.service.OperationMode
import ru.yandex.partner.core.entity.block.type.designtemplates.BlockWithDesignTemplatesBkDataHelper
import ru.yandex.partner.libs.bs.json.BkDataConverter

@Component
class BlockBkDataRepository(
    private val repository: TypedRepository<BaseBlock, *, *, *>,
    private val bkDataFacade: BkDataFacade<BaseBlock, Page.TPartnerPage.TBlock, *, BlockBkDictContainer>,
    private val actionPerformer: ActionPerformer,
    @BlockActionType(BlockActionsEnum.START_UPDATE)
    private val startUpdateFactories: List<BlockMultistateActionFactory<out BlockWithMultistate>>,
    private val blockModifyRepository: BlockModifyRepository
) : BkDataRepository<BaseBlock, Page.TPartnerPage.TBlock> {
    private val bkDataConverter = BkDataConverter()
    private val startUpdateByClass: Map<Class<out BaseBlock>, BlockMultistateActionFactory<out BlockWithMultistate>> =
        startUpdateFactories.associateBy { it.modelClass }

    override fun getBkData(ids: List<Long>, readOnly: Boolean): List<Page.TPartnerPage.TBlock> {
        val blocks = repository.getSafely(ShardSupport.NO_SHARD, ids, bkDataFacade.supportedTypes)

        if (!readOnly) {
            val startUpdateActions = blocks.groupBy { it.javaClass }
                .map { (blockClass, models) ->
                    startUpdateByClass[blockClass]!!.createAction(models.map { it.id })
                        as BlockAction<BlockWithMultistate>
                }

            val result = actionPerformer.doActions(
                false,
                startUpdateActions
            )

            if (!result.isCommitted) {
                throw BkDataException("Could not commit startUpdate action for ids $ids Result: $result")
            }
        }

        val (modelsWithCustomData, modelsWithGeneratedBkData) =
            blocks.partition { it is BlockWithCustomBkData && it.isCustomBkData }

        var idToData = mutableMapOf<Long, Page.TPartnerPage.TBlock>()

        val generatedBkData = bkDataFacade.generateBkData(
            models = modelsWithGeneratedBkData,
            container = BlockBkDictContainer()
        )

        modelsWithGeneratedBkData.forEachIndexed { index, model ->
            idToData[model.id] = generatedBkData[index]
        }

        modelsWithCustomData.forEach { model ->
            val parseResult = bkDataConverter
                .convertBlockJsonToProto((model as BlockWithCustomBkData).bkData)
            if (parseResult.errors.isNotEmpty()) {
                throw BkDataException("Could not parse custom bk data for block ${model.id} " +
                    "Errors: ${parseResult.errors}")
            }
            val parsedBkData = parseResult.message
            idToData[model.id] = parsedBkData.toBuilder()
                .setBlockID(model.blockId)
                .also(BlockWithDesignTemplatesBkDataHelper::fillAllowedImageTypes)
                .build()
        }

        idToData = idToData.mapValues { (id, bkData) ->
            val bkDataBuilder = bkData.toBuilder()
            val pageImpOptionsBuilder = bkDataBuilder.pageImpOptionsBuilder

            val enableList = pageImpOptionsBuilder.enableList.stream().sorted().toList()
            pageImpOptionsBuilder.clearEnable().addAllEnable(enableList)

            val disableList = pageImpOptionsBuilder.disableList.stream().sorted().toList()
            pageImpOptionsBuilder.clearDisable().addAllDisable(disableList)

            bkDataBuilder.build()
        }.toMutableMap()

        if (!readOnly) {
            val bkDataUpdates = blocks.asSequence()
                .filterIsInstance(BlockWithCustomBkData::class.java)
                .filter { !it.isCustomBkData }
                .map {
                    val blockData = bkDataConverter.convertProtoToJson(idToData[it.id])
                    ModelChanges.build(it.id, it.javaClass, BlockWithCustomBkData.BK_DATA, blockData)
                        .applyTo(it)
                }
                .toList()

            blockModifyRepository.update(BlockContainerImpl.create(OperationMode.EDIT), bkDataUpdates)
        }

        return ids.map(idToData::getValue)
    }
}
