package ru.yandex.partner.core.entity.block.multistate;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithSiteVersion;
import ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType;
import ru.yandex.partner.core.entity.page.model.PageWithAllowedAmp;
import ru.yandex.partner.core.entity.page.model.PageWithAllowedTurbo;
import ru.yandex.partner.core.entity.page.model.PageWithMultistate;
import ru.yandex.partner.core.multistate.page.PageStateFlag;

public class BlockCheckHelper {
    private static final Set<String> TURBO_VERSIONS = Set.of(
            SiteVersionType.TURBO.getLiteral(), SiteVersionType.TURBO_DESKTOP.getLiteral());
    private static final String AMP_VERSION = SiteVersionType.AMP.getLiteral();

    private BlockCheckHelper() {
    }

    public static <B extends BaseBlock, P extends PageWithMultistate>
    List<Boolean> isPageNotProtected(List<B> blocks, Function<B, P> pageFunction) {
        return blocks.stream()
                .map(block -> Objects.requireNonNull(pageFunction.apply(block),
                        "'Page' not loaded for block. id = " + block.getId()))
                .map(PageWithMultistate::getMultistate)
                .map(multistate -> !multistate.test(PageStateFlag.PROTECTED))
                .collect(Collectors.toList());
    }

    public static <B extends BlockWithSiteVersion, P1 extends PageWithAllowedTurbo, P2 extends PageWithAllowedAmp>
    List<Boolean> isValidSiteVersion(
            List<B> blocks, Function<B, P1> turboPageFunction, Function<B, P2> ampPageFunction) {
        return blocks.stream()
                .map(block -> !isInvalidTurboSiteVersion(block, turboPageFunction)
                        && !isInvalidAmpSiteVersion(block, ampPageFunction))
                .toList();
    }

    public static <B extends BlockWithSiteVersion, P extends PageWithAllowedTurbo>
    List<Boolean> isValidTurboSiteVersion(List<B> blocks, Function<B, P> pageFunction) {
        return blocks.stream()
                .map(block -> !isInvalidTurboSiteVersion(block, pageFunction))
                .toList();
    }

    private static <B extends BlockWithSiteVersion, P extends PageWithAllowedTurbo>
    boolean isInvalidTurboSiteVersion(B block, Function<B, P> pageFunction) {
        String siteVersion = Objects.requireNonNull(block.getSiteVersion(),
                "'SiteVersion' not loaded for block. id = " + block.getId());
        P page = Objects.requireNonNull(pageFunction.apply(block),
                "'Page' not loaded for block. id = " + block.getId());

        return isInvalidTurboSiteVersion(
                Objects.requireNonNull(page.getAllowedTurbo(),
                        "'AllowedTurbo' not loaded for block. id = " + block.getId()),
                siteVersion);
    }

    private static boolean isInvalidTurboSiteVersion(boolean allowedTurbo, String siteVersion) {
        return !allowedTurbo && TURBO_VERSIONS.contains(siteVersion);
    }

    private static <B extends BlockWithSiteVersion, P extends PageWithAllowedAmp>
    boolean isInvalidAmpSiteVersion(B block, Function<B, P> pageFunction) {
        String siteVersion = Objects.requireNonNull(block.getSiteVersion(),
                "'SiteVersion' not loaded for block. id = " + block.getId());
        P page = Objects.requireNonNull(pageFunction.apply(block),
                "'page' not loaded for block. id = " + block.getId());

        return isInvalidAmpSiteVersion(
                Objects.requireNonNull(page.getAllowedAmp(),
                        "'AllowedTurbo' not loaded for block. id = " + block.getId()),
                siteVersion);
    }

    private static boolean isInvalidAmpSiteVersion(boolean allowedAmp, String siteVersion) {
        return !allowedAmp && AMP_VERSION.contains(siteVersion);
    }

    public static <B extends BaseBlock, P extends PageWithMultistate>
    List<Boolean> isPageNotBlocked(List<B> blocks, Function<B, P> pageFunction) {
        return blocks.stream()
                .map(block -> Objects.requireNonNull(pageFunction.apply(block),
                        "'Page' not loaded for block. id = " + block.getId()))
                .map(PageWithMultistate::getMultistate)
                .map(multistate -> !multistate.test(PageStateFlag.BLOCKED))
                .collect(Collectors.toList());
    }

    public static <B extends BaseBlock, P extends PageWithMultistate>
    List<Boolean> isPageNotRejected(List<B> blocks, Function<B, P> pageFunction) {
        return blocks.stream()
                .map(block -> Objects.requireNonNull(pageFunction.apply(block),
                        "'Page' not loaded for block. id = " + block.getId()))
                .map(PageWithMultistate::getMultistate)
                .map(multistate -> !multistate.test(PageStateFlag.REJECTED))
                .collect(Collectors.toList());
    }
}
