package ru.yandex.partner.core.entity.block.multistate;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.block.model.InternalMobileRtbBlock;
import ru.yandex.partner.core.entity.page.actions.PageActionsEnum;
import ru.yandex.partner.core.entity.page.model.InternalMobileApp;
import ru.yandex.partner.core.entity.page.multistate.InternalMobileAppMultistateGraph;
import ru.yandex.partner.libs.multistate.action.ActionCheck;
import ru.yandex.partner.libs.multistate.action.ActionCheckId;
import ru.yandex.partner.libs.multistate.action.ActionCheckService;


@Component
@ParametersAreNonnullByDefault
public class InternalMobileRtbBlockActionChecksService implements ActionCheckService<InternalMobileRtbBlock> {

    private final InternalMobileAppMultistateGraph internalMobileAppMultistateGraph;

    private final Map<InternalMobileRtbBlockActionChecksService.BlockActionCheck, ActionCheck<InternalMobileRtbBlock>>
            checksMap = Map.of(
            InternalMobileRtbBlockActionChecksService.BlockActionCheck.PAGE_NOT_PROTECTED, new ActionCheck<>(
                    blocks -> BlockCheckHelper.isPageNotProtected(blocks, InternalMobileRtbBlock::getCampaign),
                    Set.of(InternalMobileRtbBlock.CAMPAIGN)),

            InternalMobileRtbBlockActionChecksService.BlockActionCheck.PAGE_ALLOWS_START_BLOCK,
            new ActionCheck<InternalMobileRtbBlock>(this::checkPageAllowsStartBlock,
                    Set.of(InternalMobileRtbBlock.CAMPAIGN)),

            InternalMobileRtbBlockActionChecksService.BlockActionCheck.PAGE_ALLOWS_RESTORE_BLOCK,
            new ActionCheck<InternalMobileRtbBlock>(this::checkPageAllowsRestoreBlock,
                    Set.of(InternalMobileRtbBlock.CAMPAIGN)),

            InternalMobileRtbBlockActionChecksService.BlockActionCheck.BLOCKS_LIMIT_NOT_EXCEEDED,
            new ActionCheck<InternalMobileRtbBlock>(
                    this::checkBlocksLimitNotExceeded,
                    Set.of(InternalMobileRtbBlock.CAMPAIGN)
            )
    );

    @Autowired
    public InternalMobileRtbBlockActionChecksService(
            InternalMobileAppMultistateGraph internalContextPageMultistateGraph) {
        this.internalMobileAppMultistateGraph = internalContextPageMultistateGraph;
    }

    @Override
    public ActionCheck<InternalMobileRtbBlock> getActionCheck(ActionCheckId id) {
        if (!checksMap.containsKey(id)) {
            throw new IllegalArgumentException(String.format("Action check with id '%s' not found", id));
        }

        return checksMap.get(id);
    }

    private List<Boolean> checkPageAllowsStartBlock(List<? extends InternalMobileRtbBlock> models) {
        List<InternalMobileApp> pages = models.stream().map(InternalMobileRtbBlock::getCampaign).toList();
        return internalMobileAppMultistateGraph.checkActionAllowed(
                PageActionsEnum.START_BLOCK.getActionName(),
                pages
        );
    }

    private List<Boolean> checkPageAllowsRestoreBlock(List<? extends InternalMobileRtbBlock> models) {
        List<InternalMobileApp> pages = models.stream().map(InternalMobileRtbBlock::getCampaign).toList();
        return internalMobileAppMultistateGraph.checkActionAllowed(
                PageActionsEnum.RESTORE_BLOCK.getActionName(),
                pages
        );
    }

    public List<Boolean> checkBlocksLimitNotExceeded(List<? extends InternalMobileRtbBlock> models) {
        return models.stream()
                .map(InternalMobileRtbBlock::getCampaign)
                .map(ActionChecksUtils::isBlocksLimitNotExceeded)
                .toList();
    }

    public enum BlockActionCheck implements ActionCheckId {
        PAGE_NOT_PROTECTED,
        PAGE_ALLOWS_START_BLOCK,
        PAGE_ALLOWS_RESTORE_BLOCK,
        BLOCKS_LIMIT_NOT_EXCEEDED
    }
}
