package ru.yandex.partner.core.entity.block.multistate;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.block.actions.BlockActionsEnum;
import ru.yandex.partner.core.entity.block.model.InternalMobileRtbBlock;
import ru.yandex.partner.core.messages.BlockActionMsg;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.block.BlockMultistate;
import ru.yandex.partner.core.multistate.block.BlockStateFlag;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.multistate.action.ActionCheckId;
import ru.yandex.partner.libs.multistate.action.ActionEntry;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.AbstractMultistateGraph;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.empty;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.hasNoneOf;

@Component
@ParametersAreNonnullByDefault
public class InternalMobileRtbBlockMultistateGraph
        extends AbstractMultistateGraph<InternalMobileRtbBlock, BlockStateFlag> {

    private final InternalMobileRtbBlockActionChecksService blockActionChecksService;

    @Autowired
    public InternalMobileRtbBlockMultistateGraph(InternalMobileRtbBlockActionChecksService blockActionChecksService) {
        this.blockActionChecksService = blockActionChecksService;
    }

    @Override
    protected List<Boolean> performCheck(ActionCheckId check, List<? extends InternalMobileRtbBlock> models) {
        if (!(check instanceof InternalMobileRtbBlockActionChecksService.BlockActionCheck)) {
            return super.performCheck(check, models);
        }
        Function<List<? extends InternalMobileRtbBlock>, List<Boolean>> checkFunction =
                blockActionChecksService.getActionCheck(check).getCheck();
        return checkFunction == null ? super.performCheck(check, models)
                : checkFunction.apply(models);
    }

    @Override
    public Set<ModelProperty<? super InternalMobileRtbBlock, ?>> getAllRequiredProperties() {
        return this.getAllAvailableActionNames().stream()
                .map(this::getRequiredPropertiesByActionName)
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
    }

    @Override
    public Set<ModelProperty<? super InternalMobileRtbBlock, ?>> getRequiredPropertiesByActionName(String actionName) {
        var actionEntry = this.getActionEntry(actionName);

        Set<ModelProperty<? super InternalMobileRtbBlock, ?>> requiredProperties =
                new HashSet<>(actionEntry.getRequiredProperties());

        var requiredPropertiesFromChecks = actionEntry.getChecks().stream()
                .map(check -> blockActionChecksService.getActionCheck(check).getRequiredModelProperties())
                .flatMap(Set::stream).collect(Collectors.toSet());

        requiredProperties.addAll(requiredPropertiesFromChecks);

        return requiredProperties;
    }

    @Override
    protected Multistate<BlockStateFlag> getMultistateForValue(Long multistateValue) {
        return new BlockMultistate(multistateValue);
    }

    @Override
    public Multistate<BlockStateFlag> getMultistateFromModel(InternalMobileRtbBlock model) {
        return model.getMultistate();
    }

    @Override
    public Class<InternalMobileRtbBlock> getModelClass() {
        return InternalMobileRtbBlock.class;
    }

    @Override
    public Multistate<BlockStateFlag> convertMultistate(List<BlockStateFlag> enabledFlags) {
        return new BlockMultistate(enabledFlags);
    }

    private ActionEntry.Builder<InternalMobileRtbBlock, BlockStateFlag> getActionEntryBuilder(GettextMsg titleMsg) {
        return getActionEntryBuilder(titleMsg, Set.of(InternalMobileRtbBlock.ID, InternalMobileRtbBlock.MULTISTATE));
    }

    @Override
    protected Map<ActionNameHolder, ActionEntry<InternalMobileRtbBlock, BlockStateFlag>> createGraph() {
        Map<ActionNameHolder, ActionEntry<InternalMobileRtbBlock, BlockStateFlag>> actionEntryMap = new HashMap<>();
        actionEntryMap.put(BlockActionsEnum.ADD,
                getActionEntryBuilder(BlockActionMsg.ADD)
                        .setPredicate(empty())
                        .build());

        actionEntryMap.put(BlockActionsEnum.DELETE,
                getActionEntryBuilder(BlockActionMsg.ARCHIVE)
                        .setPredicate(not(BlockStateFlag.DELETED))
                        .setSetFlags(BlockStateFlag.DELETED, true)
                        .build());

        actionEntryMap.put(BlockActionsEnum.DELETE_WITH_PAGE,
                getActionEntryBuilder(BlockActionMsg.ARCHIVE_WITH_PAGE)
                        .setPredicate(not(BlockStateFlag.DELETED))
                        .setSetFlags(Map.of(BlockStateFlag.DELETED, true,
                                BlockStateFlag.DELETED_WITH_PAGE, true))
                        .build());

        actionEntryMap.put(BlockActionsEnum.RESTORE,
                getActionEntryBuilder(BlockActionMsg.RESTORE)
                        .setPredicate(
                                has(BlockStateFlag.DELETED).and(not(BlockStateFlag.DELETED_WITH_PAGE)))
                        .setChecks(List.of(
                                InternalMobileRtbBlockActionChecksService.BlockActionCheck.PAGE_ALLOWS_RESTORE_BLOCK,
                                InternalMobileRtbBlockActionChecksService.BlockActionCheck.BLOCKS_LIMIT_NOT_EXCEEDED))
                        .setSetFlags(BlockStateFlag.DELETED, false)
                        .build());

        actionEntryMap.put(BlockActionsEnum.RESTORE_WITH_PAGE,
                getActionEntryBuilder(BlockActionMsg.RESTORE_WITH_PAGE)
                        .setPredicate(
                                has(BlockStateFlag.DELETED).and(has(BlockStateFlag.DELETED_WITH_PAGE)))
                        .setSetFlags(Map.of(BlockStateFlag.DELETED, false,
                                BlockStateFlag.DELETED_WITH_PAGE, false))
                        .build());

        actionEntryMap.put(BlockActionsEnum.EDIT,
                getActionEntryBuilder(BlockActionMsg.EDIT, Set.of())
                        .setPredicate(not(BlockStateFlag.DELETED))
                        .build());

        actionEntryMap.put(BlockActionsEnum.START,
                getActionEntryBuilder(BlockActionMsg.START)
                        .setPredicate(hasNoneOf(BlockStateFlag.DELETED, BlockStateFlag.WORKING))
                        .setChecks(List.of(
                                InternalMobileRtbBlockActionChecksService.BlockActionCheck.PAGE_ALLOWS_START_BLOCK))
                        .setSetFlags(BlockStateFlag.WORKING, true)
                        .build());

        actionEntryMap.put(BlockActionsEnum.STOP,
                getActionEntryBuilder(BlockActionMsg.STOP)
                        .setPredicate(has(BlockStateFlag.WORKING))
                        .setSetFlags(BlockStateFlag.WORKING, false)
                        .build());

        actionEntryMap.put(BlockActionsEnum.SET_CHECK_STATISTICS,
                getActionEntryBuilder(BlockActionMsg.SET_CHECK_STATISTICS)
                        .setPredicate(not(BlockStateFlag.DELETED))
                        .setSetFlags(BlockStateFlag.CHECK_STATISTICS, true)
                        .build());

        actionEntryMap.put(BlockActionsEnum.RESET_CHECK_STATISTICS,
                getActionEntryBuilder(BlockActionMsg.RESET_CHECK_STATISTICS)
                        .setPredicate(has(BlockStateFlag.CHECK_STATISTICS))
                        .setSetFlags(BlockStateFlag.CHECK_STATISTICS, false)
                        .build());

        actionEntryMap.put(BlockActionsEnum.DUPLICATE,
                getActionEntryBuilder(BlockActionMsg.DUPLICATE)
                        .setPredicate(hasNoneOf(BlockStateFlag.DELETED, BlockStateFlag.DELETED_WITH_PAGE))
                        .setChecks(List.of(
                                InternalMobileRtbBlockActionChecksService.BlockActionCheck.PAGE_NOT_PROTECTED,
                                InternalMobileRtbBlockActionChecksService.BlockActionCheck.BLOCKS_LIMIT_NOT_EXCEEDED))
                        .build());

        actionEntryMap.put(BlockActionsEnum.SET_NEED_UPDATE,
                getActionEntryBuilder(BlockActionMsg.SET_NEED_UPDATE)
                        .setSetFlags(BlockStateFlag.NEED_UPDATE, true)
                        .build());

        actionEntryMap.put(BlockActionsEnum.START_UPDATE,
                getActionEntryBuilder(BlockActionMsg.START_UPDATE)
                        .setPredicate(has(BlockStateFlag.NEED_UPDATE))
                        .setSetFlags(Map.of(
                                BlockStateFlag.UPDATING, true,
                                BlockStateFlag.NEED_UPDATE, false))
                        .build());

        actionEntryMap.put(BlockActionsEnum.STOP_UPDATE,
                getActionEntryBuilder(BlockActionMsg.STOP_UPDATE)
                        .setPredicate(has(BlockStateFlag.UPDATING))
                        .setSetFlags(BlockStateFlag.UPDATING, false)
                        .build());

        return actionEntryMap;
    }
}
