package ru.yandex.partner.core.entity.block.multistate;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.block.model.BlockWithAdfox;
import ru.yandex.partner.core.entity.block.model.BlockWithInternalContextPage;
import ru.yandex.partner.core.entity.block.model.BlockWithInternalContextPageAndSiteVersion;
import ru.yandex.partner.core.entity.block.model.BlockWithSiteVersion;
import ru.yandex.partner.core.entity.block.model.InternalRtbBlock;
import ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType;
import ru.yandex.partner.core.entity.block.type.siteversionandcontextpage.SiteVersionAvailabilityService;
import ru.yandex.partner.core.entity.page.actions.PageActionsEnum;
import ru.yandex.partner.core.entity.page.model.InternalContextPage;
import ru.yandex.partner.core.entity.page.multistate.InternalContextPageMultistateGraph;
import ru.yandex.partner.libs.multistate.action.ActionCheck;
import ru.yandex.partner.libs.multistate.action.ActionCheckId;
import ru.yandex.partner.libs.multistate.action.ActionCheckService;

import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_FLOORAD;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_FULLSCREEN;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_REWARDED;


@Component
@ParametersAreNonnullByDefault
public class InternalRtbBlockActionChecksService implements ActionCheckService<InternalRtbBlock> {
    private final InternalContextPageMultistateGraph internalContextPageMultistateGraph;
    private final SiteVersionAvailabilityService siteVersionAvailabilityService;

    private final Map<BlockActionCheck, ActionCheck<InternalRtbBlock>> checksMap = Map.of(
            BlockActionCheck.PAGE_NOT_PROTECTED, new ActionCheck<>(
                    blocks -> BlockCheckHelper.isPageNotProtected(blocks, InternalRtbBlock::getCampaign),
                    Set.of(BlockWithInternalContextPage.CAMPAIGN)),

            BlockActionCheck.VALID_SITE_VERSION, new ActionCheck<>(
                    blocks -> BlockCheckHelper.isValidTurboSiteVersion(blocks, InternalRtbBlock::getCampaign),
                    Set.of(BlockWithInternalContextPage.CAMPAIGN, BlockWithSiteVersion.SITE_VERSION)),

            BlockActionCheck.PAGE_ALLOWS_START_BLOCK, new ActionCheck<InternalRtbBlock>(this::checkPageAllowsStartBlock,
                    Set.of(BlockWithInternalContextPage.CAMPAIGN)),

            BlockActionCheck.PAGE_ALLOWS_RESTORE_BLOCK,
            new ActionCheck<InternalRtbBlock>(this::checkPageAllowsRestoreBlock,
                    Set.of(BlockWithInternalContextPage.CAMPAIGN)),

            BlockActionCheck.IS_ADFOX_BLOCK,
            new ActionCheck<InternalRtbBlock>(this::checkIsAdfoxBlock,
                    Set.of(BlockWithAdfox.ADFOX_BLOCK)),

            BlockActionCheck.NOT_ADFOX_BLOCK, new ActionCheck<InternalRtbBlock>(this::checkIsNotAdfoxBlock,
                    Set.of(BlockWithAdfox.ADFOX_BLOCK)),
            BlockActionCheck.HAS_SITE_VERSION_FEATURE,
            new ActionCheck<InternalRtbBlock>(this::checkFeature,
                    Set.of(BlockWithInternalContextPageAndSiteVersion.CAMPAIGN,
                            BlockWithInternalContextPageAndSiteVersion.SITE_VERSION))
    );

    @Autowired
    public InternalRtbBlockActionChecksService(InternalContextPageMultistateGraph internalContextPageMultistateGraph,
                                               SiteVersionAvailabilityService siteVersionAvailabilityService) {
        this.internalContextPageMultistateGraph = internalContextPageMultistateGraph;
        this.siteVersionAvailabilityService = siteVersionAvailabilityService;
    }

    @Override
    public ActionCheck<InternalRtbBlock> getActionCheck(ActionCheckId id) {
        if (!checksMap.containsKey(id)) {
            throw new IllegalArgumentException(String.format("Action check with id '%s' not found", id));
        }

        return checksMap.get(id);
    }

    private List<Boolean> checkIsAdfoxBlock(List<? extends BlockWithAdfox> models) {
        return models.stream().map(model -> Boolean.TRUE.equals(model.getAdfoxBlock())).collect(Collectors.toList());
    }

    private List<Boolean> checkIsNotAdfoxBlock(List<? extends BlockWithAdfox> models) {
        return models.stream().map(model -> Boolean.FALSE.equals(model.getAdfoxBlock())).collect(Collectors.toList());
    }

    public List<Boolean> checkFeature(List<? extends BlockWithInternalContextPageAndSiteVersion> models) {
        var mobileSiteVersions = SiteVersionType.literals(MOBILE_FULLSCREEN, MOBILE_REWARDED, MOBILE_FLOORAD);

        return models.stream()
                .map(block -> {
                    if (mobileSiteVersions.contains(block.getSiteVersion())) {
                        return siteVersionAvailabilityService.getSiteVersionsThatDependOnFeature(block.getClass(),
                                        new HashSet<>(block.getCampaign().getOwner().getFeatures()))
                                .contains(block.getSiteVersion());
                    } else {
                        return true;
                    }
                }).toList();
    }

    private List<Boolean> checkPageAllowsStartBlock(List<? extends InternalRtbBlock> models) {
        List<InternalContextPage> pages = models.stream().map(InternalRtbBlock::getCampaign).toList();
        return internalContextPageMultistateGraph.checkActionAllowed(
                PageActionsEnum.START_BLOCK.getActionName(),
                pages
        );
    }

    private List<Boolean> checkPageAllowsRestoreBlock(List<? extends InternalRtbBlock> models) {
        List<InternalContextPage> pages = models.stream().map(InternalRtbBlock::getCampaign).toList();
        return internalContextPageMultistateGraph.checkActionAllowed(
                PageActionsEnum.RESTORE_BLOCK.getActionName(),
                pages
        );
    }

    public enum BlockActionCheck implements ActionCheckId {
        PAGE_NOT_PROTECTED,
        VALID_SITE_VERSION,
        PAGE_ALLOWS_START_BLOCK,
        PAGE_ALLOWS_RESTORE_BLOCK,
        IS_ADFOX_BLOCK,
        NOT_ADFOX_BLOCK,
        HAS_SITE_VERSION_FEATURE
    }
}
