package ru.yandex.partner.core.entity.block.multistate;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.block.actions.BlockActionsEnum;
import ru.yandex.partner.core.entity.block.model.InternalRtbBlock;
import ru.yandex.partner.core.messages.BlockActionMsg;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.block.BlockMultistate;
import ru.yandex.partner.core.multistate.block.BlockStateFlag;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.multistate.action.ActionCheckId;
import ru.yandex.partner.libs.multistate.action.ActionEntry;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.AbstractMultistateGraph;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.core.multistate.block.BlockStateFlag.DELETED;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.empty;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.hasNoneOf;

@Component
@ParametersAreNonnullByDefault
public class InternalRtbBlockMultistateGraph extends AbstractMultistateGraph<InternalRtbBlock, BlockStateFlag> {

    private final InternalRtbBlockActionChecksService blockActionChecksService;

    @Autowired
    public InternalRtbBlockMultistateGraph(InternalRtbBlockActionChecksService blockActionChecksService) {
        this.blockActionChecksService = blockActionChecksService;
    }

    @Override
    protected List<Boolean> performCheck(ActionCheckId check, List<? extends InternalRtbBlock> models) {
        if (!(check instanceof InternalRtbBlockActionChecksService.BlockActionCheck)) {
            return super.performCheck(check, models);
        }
        Function<List<? extends InternalRtbBlock>, List<Boolean>> checkFunction =
                blockActionChecksService.getActionCheck(check).getCheck();
        return checkFunction == null ? super.performCheck(check, models)
                : checkFunction.apply(models);
    }

    @Override
    public Set<ModelProperty<? super InternalRtbBlock, ?>> getAllRequiredProperties() {
        return this.getAllAvailableActionNames().stream()
                .map(this::getRequiredPropertiesByActionName)
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
    }

    @Override
    public Set<ModelProperty<? super InternalRtbBlock, ?>> getRequiredPropertiesByActionName(String actionName) {
        var actionEntry = this.getActionEntry(actionName);

        Set<ModelProperty<? super InternalRtbBlock, ?>> requiredProperties =
                new HashSet<>(actionEntry.getRequiredProperties());

        var requiredPropertiesFromChecks = actionEntry.getChecks().stream()
                .map(check -> blockActionChecksService.getActionCheck(check).getRequiredModelProperties())
                .flatMap(Set::stream).collect(Collectors.toSet());

        requiredProperties.addAll(requiredPropertiesFromChecks);

        return requiredProperties;
    }

    @Override
    protected Multistate<BlockStateFlag> getMultistateForValue(Long multistateValue) {
        return new BlockMultistate(multistateValue);
    }

    @Override
    public Multistate<BlockStateFlag> getMultistateFromModel(InternalRtbBlock model) {
        return model.getMultistate();
    }

    @Override
    public Class<InternalRtbBlock> getModelClass() {
        return InternalRtbBlock.class;
    }

    @Override
    public Multistate<BlockStateFlag> convertMultistate(List<BlockStateFlag> enabledFlags) {
        return new BlockMultistate(enabledFlags);
    }

    private ActionEntry.Builder<InternalRtbBlock, BlockStateFlag> getActionEntryBuilder(GettextMsg titleMsg) {
        return getActionEntryBuilder(titleMsg, Set.of(InternalRtbBlock.ID, InternalRtbBlock.MULTISTATE));
    }

    @Override
    protected Map<ActionNameHolder, ActionEntry<InternalRtbBlock, BlockStateFlag>> createGraph() {
        Map<ActionNameHolder, ActionEntry<InternalRtbBlock, BlockStateFlag>> actionEntryMap = new HashMap<>();
        actionEntryMap.put(BlockActionsEnum.ADD,
                getActionEntryBuilder(BlockActionMsg.ADD)
                        .setPredicate(empty())
                        .build());

        actionEntryMap.put(BlockActionsEnum.DELETE,
                getActionEntryBuilder(BlockActionMsg.ARCHIVE)
                        .setPredicate(not(has(BlockStateFlag.DELETED)))
                        .setChecks(
                                List.of(
                                        InternalRtbBlockActionChecksService.BlockActionCheck.NOT_ADFOX_BLOCK
                                )
                        )
                        .setSetFlags(BlockStateFlag.DELETED, true)
                        .build());

        actionEntryMap.put(BlockActionsEnum.DELETE_WITH_PAGE,
                getActionEntryBuilder(BlockActionMsg.ARCHIVE_WITH_PAGE)
                        .setPredicate(not(has(BlockStateFlag.DELETED)))
                        .setSetFlags(Map.of(BlockStateFlag.DELETED, true,
                                BlockStateFlag.DELETED_WITH_PAGE, true))
                        .build());

        actionEntryMap.put(BlockActionsEnum.RESTORE,
                getActionEntryBuilder(BlockActionMsg.RESTORE)
                        .setPredicate(
                                has(BlockStateFlag.DELETED).and(not(has(BlockStateFlag.DELETED_WITH_PAGE))))
                        .setSetFlags(BlockStateFlag.DELETED, false)
                        .setChecks(List.of(
                                InternalRtbBlockActionChecksService.BlockActionCheck.VALID_SITE_VERSION,
                                InternalRtbBlockActionChecksService.BlockActionCheck.PAGE_ALLOWS_RESTORE_BLOCK,
                                InternalRtbBlockActionChecksService.BlockActionCheck.HAS_SITE_VERSION_FEATURE
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.RESTORE_WITH_PAGE,
                getActionEntryBuilder(BlockActionMsg.RESTORE_WITH_PAGE)
                        .setPredicate(
                                has(BlockStateFlag.DELETED).and(has(BlockStateFlag.DELETED_WITH_PAGE)))
                        .setSetFlags(Map.of(BlockStateFlag.DELETED, false,
                                BlockStateFlag.DELETED_WITH_PAGE, false))
                        .setChecks(List.of(InternalRtbBlockActionChecksService.BlockActionCheck.VALID_SITE_VERSION))
                        .build());

        actionEntryMap.put(BlockActionsEnum.EDIT,
                getActionEntryBuilder(BlockActionMsg.EDIT, Set.of())
                        .setPredicate(not(has(BlockStateFlag.DELETED)))
                        .build());

        actionEntryMap.put(BlockActionsEnum.START,
                getActionEntryBuilder(BlockActionMsg.START)
                        .setPredicate(hasNoneOf(BlockStateFlag.DELETED, BlockStateFlag.WORKING))
                        .setSetFlags(BlockStateFlag.WORKING, true)
                        .setChecks(
                                List.of(InternalRtbBlockActionChecksService.BlockActionCheck.PAGE_ALLOWS_START_BLOCK))
                        .build());

        actionEntryMap.put(BlockActionsEnum.STOP,
                getActionEntryBuilder(BlockActionMsg.STOP)
                        .setPredicate(has(BlockStateFlag.WORKING))
                        .setSetFlags(BlockStateFlag.WORKING, false)
                        .build());

        actionEntryMap.put(BlockActionsEnum.SET_CHECK_STATISTICS,
                getActionEntryBuilder(BlockActionMsg.SET_CHECK_STATISTICS)
                        .setPredicate(not(has(BlockStateFlag.DELETED)))
                        .setSetFlags(BlockStateFlag.CHECK_STATISTICS, true)
                        .build());

        actionEntryMap.put(BlockActionsEnum.RESET_CHECK_STATISTICS,
                getActionEntryBuilder(BlockActionMsg.RESET_CHECK_STATISTICS)
                        .setPredicate(has(BlockStateFlag.CHECK_STATISTICS))
                        .setSetFlags(BlockStateFlag.CHECK_STATISTICS, false)
                        .build());


        actionEntryMap.put(BlockActionsEnum.DUPLICATE,
                getActionEntryBuilder(BlockActionMsg.DUPLICATE)
                        .setPredicate(hasNoneOf(BlockStateFlag.DELETED, BlockStateFlag.DELETED_WITH_PAGE))
                        .setChecks(
                                List.of(
                                        InternalRtbBlockActionChecksService.BlockActionCheck.PAGE_NOT_PROTECTED,
                                        InternalRtbBlockActionChecksService.BlockActionCheck.NOT_ADFOX_BLOCK
                                )
                        )
                        .build());

        actionEntryMap.put(BlockActionsEnum.SET_NEED_UPDATE,
                getActionEntryBuilder(BlockActionMsg.SET_NEED_UPDATE)
                        .setSetFlags(BlockStateFlag.NEED_UPDATE, true)
                        .build());

        actionEntryMap.put(BlockActionsEnum.START_UPDATE,
                getActionEntryBuilder(BlockActionMsg.START_UPDATE)
                        .setPredicate(has(BlockStateFlag.NEED_UPDATE))
                        .setSetFlags(Map.of(
                                BlockStateFlag.UPDATING, true,
                                BlockStateFlag.NEED_UPDATE, false))
                        .build());

        actionEntryMap.put(BlockActionsEnum.STOP_UPDATE,
                getActionEntryBuilder(BlockActionMsg.STOP_UPDATE)
                        .setPredicate(has(BlockStateFlag.UPDATING))
                        .setSetFlags(BlockStateFlag.UPDATING, false)
                        .build());

        actionEntryMap.put(BlockActionsEnum.DELETE_FROM_ADFOX,
                getActionEntryBuilder(BlockActionMsg.ARCHIVE)
                        .setPredicate(not(has(DELETED)))
                        .setChecks(List.of(
                                InternalRtbBlockActionChecksService.BlockActionCheck.IS_ADFOX_BLOCK
                        ))
                        .setSetFlags(DELETED, true)
                        .build());

        return actionEntryMap;
    }
}
