package ru.yandex.partner.core.entity.block.multistate;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.block.model.BlockWithAdfox;
import ru.yandex.partner.core.entity.block.model.BlockWithMobileAppSettings;
import ru.yandex.partner.core.entity.block.model.MobileRtbBlock;
import ru.yandex.partner.core.entity.page.actions.PageActionsEnum;
import ru.yandex.partner.core.entity.page.model.MobileAppSettings;
import ru.yandex.partner.core.entity.page.multistate.MobileAppMultistateGraph;
import ru.yandex.partner.libs.multistate.action.ActionCheck;
import ru.yandex.partner.libs.multistate.action.ActionCheckId;
import ru.yandex.partner.libs.multistate.action.ActionCheckService;

@Component
@ParametersAreNonnullByDefault
public class MobileRtbBlockActionChecksService implements ActionCheckService<MobileRtbBlock> {

    private final MobileAppMultistateGraph mobileAppMultistateGraph;

    private final Map<BlockActionCheck, ActionCheck<MobileRtbBlock>> checksMap = Map.of(

            BlockActionCheck.PAGE_NOT_PROTECTED,
            new ActionCheck<>(
                    blocks -> BlockCheckHelper.isPageNotProtected(blocks, MobileRtbBlock::getCampaign),
                    Set.of(BlockWithMobileAppSettings.CAMPAIGN)),

            BlockActionCheck.NOT_ADFOX_BLOCK,
            new ActionCheck<MobileRtbBlock>(this::checkIsNotAdfoxBlock,
                    Set.of(BlockWithAdfox.ADFOX_BLOCK)),

            BlockActionCheck.PAGE_ALLOWS_RESTORE_BLOCK,
            new ActionCheck<MobileRtbBlock>(this::checkPageAllowsRestoreBlock,
                    Set.of(BlockWithMobileAppSettings.CAMPAIGN)),

            BlockActionCheck.PAGE_NOT_BLOCKED,
            new ActionCheck<>(
                    blocks -> BlockCheckHelper.isPageNotBlocked(blocks, MobileRtbBlock::getCampaign),
                    Set.of(BlockWithMobileAppSettings.CAMPAIGN)),

            BlockActionCheck.PAGE_NOT_REJECTED,
            new ActionCheck<>(
                    blocks -> BlockCheckHelper.isPageNotRejected(blocks, MobileRtbBlock::getCampaign),
                    Set.of(BlockWithMobileAppSettings.CAMPAIGN)),

            BlockActionCheck.BLOCKS_LIMIT_NOT_EXCEEDED, new ActionCheck<MobileRtbBlock>(
                    this::checkBlocksLimitNotExceeded,
                    Set.of(BlockWithMobileAppSettings.CAMPAIGN)
            )
    );

    @Autowired
    public MobileRtbBlockActionChecksService(MobileAppMultistateGraph mobileAppMultistateGraph) {
        this.mobileAppMultistateGraph = mobileAppMultistateGraph;
    }

    @Override
    public ActionCheck<MobileRtbBlock> getActionCheck(ActionCheckId id) {
        if (!checksMap.containsKey(id)) {
            throw new IllegalArgumentException(String.format("Action check with id '%s' not found", id));
        }

        return checksMap.get(id);
    }

    private List<Boolean> checkIsNotAdfoxBlock(List<? extends BlockWithAdfox> models) {
        return models.stream().map(model -> Boolean.FALSE.equals(model.getAdfoxBlock())).collect(Collectors.toList());
    }

    private List<Boolean> checkPageAllowsRestoreBlock(List<? extends MobileRtbBlock> models) {
        List<MobileAppSettings> pages = models.stream().map(MobileRtbBlock::getCampaign).toList();
        return mobileAppMultistateGraph.checkActionAllowed(
                PageActionsEnum.RESTORE_BLOCK.getActionName(),
                pages
        );
    }

    public List<Boolean> checkBlocksLimitNotExceeded(List<? extends MobileRtbBlock> models) {
        return models.stream()
                .map(MobileRtbBlock::getCampaign)
                .map(ActionChecksUtils::isBlocksLimitNotExceeded)
                .toList();
    }

    public enum BlockActionCheck implements ActionCheckId {
        PAGE_NOT_PROTECTED,
        PAGE_ALLOWS_RESTORE_BLOCK,
        NOT_ADFOX_BLOCK,
        PAGE_NOT_BLOCKED,
        PAGE_NOT_REJECTED,
        BLOCKS_LIMIT_NOT_EXCEEDED
    }
}
