package ru.yandex.partner.core.entity.block.multistate;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import io.micrometer.core.lang.NonNullApi;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithAdfox;
import ru.yandex.partner.core.entity.block.model.BlockWithContextPage;
import ru.yandex.partner.core.entity.block.model.BlockWithSiteVersionAndContextPage;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType;
import ru.yandex.partner.core.entity.block.type.siteversionandcontextpage.SiteVersionAvailabilityService;
import ru.yandex.partner.core.entity.page.actions.PageActionsEnum;
import ru.yandex.partner.core.entity.page.model.ContextPage;
import ru.yandex.partner.core.entity.page.multistate.ContextPageMultistateGraph;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.libs.multistate.action.ActionCheck;
import ru.yandex.partner.libs.multistate.action.ActionCheckId;
import ru.yandex.partner.libs.multistate.action.ActionCheckService;

import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_FLOORAD;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_FULLSCREEN;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_REWARDED;

@Service
@NonNullApi
public class RtbBlockActionChecksService implements ActionCheckService<RtbBlock> {
    private final ContextPageMultistateGraph contextPageMultistateGraph;
    private final SiteVersionAvailabilityService siteVersionAvailabilityService;

    private final Map<BlockActionCheck, ActionCheck<RtbBlock>> checksMap = Map.of(
            BlockActionCheck.IS_ADFOX_BLOCK, new ActionCheck<RtbBlock>(this::checkIsAdfoxBlock,
                    Set.of(BlockWithAdfox.ADFOX_BLOCK)),
            BlockActionCheck.NOT_ADFOX_BLOCK, new ActionCheck<RtbBlock>(this::checkIsNotAdfoxBlock,
                    Set.of(BlockWithAdfox.ADFOX_BLOCK)),
            BlockActionCheck.PAGE_NOT_PROTECTED, new ActionCheck<>(
                    blocks -> BlockCheckHelper.isPageNotProtected(blocks, RtbBlock::getCampaign),
                    Set.of(BlockWithContextPage.CAMPAIGN)),
            BlockActionCheck.PAGE_NOT_BLOCKED, new ActionCheck<RtbBlock>(this::checkPageIsNotBlocked,
                    Set.of(BlockWithContextPage.CAMPAIGN)),
            BlockActionCheck.PAGE_NOT_REJECTED, new ActionCheck<RtbBlock>(this::checkPageIsNotRejected,
                    Set.of(BlockWithContextPage.CAMPAIGN)),
            BlockActionCheck.VALID_SITE_VERSION, new ActionCheck<>(
                    rtbBlocks -> BlockCheckHelper.isValidSiteVersion(
                            rtbBlocks, RtbBlock::getCampaign, RtbBlock::getCampaign),
                    Set.of(BlockWithSiteVersionAndContextPage.CAMPAIGN,
                            BlockWithSiteVersionAndContextPage.SITE_VERSION)),
            BlockActionCheck.PAGE_ALLOWS_START_BLOCK, new ActionCheck<RtbBlock>(this::checkPageAllowsStartBlock,
                    Set.of(BlockWithContextPage.CAMPAIGN)),
            BlockActionCheck.PAGE_ALLOWS_RESTORE_BLOCK, new ActionCheck<RtbBlock>(this::checkPageAllowsRestoreBlock,
                    Set.of(BlockWithContextPage.CAMPAIGN)),
            BlockActionCheck.BLOCKS_LIMIT_NOT_EXCEEDED, new ActionCheck<RtbBlock>(this::checkBlocksLimitNotExceeded,
                    Set.of(BlockWithContextPage.CAMPAIGN)),
            BlockActionCheck.HAS_SITE_VERSION_FEATURE, new ActionCheck<RtbBlock>(this::checkFeature,
                    Set.of(BlockWithSiteVersionAndContextPage.CAMPAIGN,
                            BlockWithSiteVersionAndContextPage.SITE_VERSION))
    );

    public RtbBlockActionChecksService(ContextPageMultistateGraph contextPageMultistateGraph,
                                       SiteVersionAvailabilityService siteVersionAvailabilityService) {
        this.contextPageMultistateGraph = contextPageMultistateGraph;
        this.siteVersionAvailabilityService = siteVersionAvailabilityService;
    }

    @Override
    public ActionCheck<RtbBlock> getActionCheck(ActionCheckId id) {
        if (!checksMap.containsKey(id)) {
            throw new IllegalArgumentException(String.format("Action check with id '%s' not found", id));
        }

        return checksMap.get(id);
    }

    public List<Boolean> checkIsAdfoxBlock(List<? extends BlockWithAdfox> models) {
        return models.stream().map(this::isAdfoxBlock).collect(Collectors.toList());
    }

    public List<Boolean> checkIsNotAdfoxBlock(List<? extends BlockWithAdfox> models) {
        return models.stream().map(model -> !isAdfoxBlock(model)).collect(Collectors.toList());
    }

    public List<Boolean> checkPageIsNotBlocked(List<? extends BaseBlock> models) {
        return models.stream()
                .map(this::getPageForBlock)
                .map(ContextPage::getMultistate)
                .map(multistate -> !multistate.test(PageStateFlag.BLOCKED))
                .collect(Collectors.toList());
    }

    public List<Boolean> checkPageIsNotRejected(List<? extends BaseBlock> models) {
        return models.stream()
                .map(this::getPageForBlock)
                .map(ContextPage::getMultistate)
                .map(multistate -> !multistate.test(PageStateFlag.REJECTED))
                .collect(Collectors.toList());
    }

    public List<Boolean> checkPageAllowsRestoreBlock(List<? extends BaseBlock> models) {
        List<ContextPage> pages = getPagesForBlocks(models);
        return contextPageMultistateGraph.checkActionAllowed(
                PageActionsEnum.RESTORE_BLOCK.getActionName(),
                pages
        );
    }

    public List<Boolean> checkPageAllowsStartBlock(List<? extends BaseBlock> models) {
        List<ContextPage> pages = getPagesForBlocks(models);
        return contextPageMultistateGraph.checkActionAllowed(
                PageActionsEnum.START_BLOCK.getActionName(),
                pages
        );
    }

    public List<Boolean> checkBlocksLimitNotExceeded(List<? extends RtbBlock> models) {
        return models.stream()
                .map(this::getPageForBlock)
                .map(ActionChecksUtils::isBlocksLimitNotExceeded)
                .collect(Collectors.toList());
    }

    public List<Boolean> checkFeature(List<? extends BlockWithSiteVersionAndContextPage> models) {
        var mobileSiteVersions = SiteVersionType.literals(MOBILE_FULLSCREEN, MOBILE_REWARDED, MOBILE_FLOORAD);

        return models.stream()
                .map(block -> {
                    if (mobileSiteVersions.contains(block.getSiteVersion())) {
                        return siteVersionAvailabilityService.getSiteVersionsThatDependOnFeature(block.getClass(),
                                        new HashSet<>(block.getCampaign().getOwner().getFeatures()))
                                .contains(block.getSiteVersion());
                    } else {
                        return true;
                    }
                }).toList();
    }

    public List<ContextPage> getPagesForBlocks(List<? extends BaseBlock> models) {
        return models.stream().map(this::getPageForBlock).collect(Collectors.toList());
    }

    private ContextPage getPageForBlock(BaseBlock model) {
        if (!(model instanceof BlockWithContextPage)) {
            throw new IllegalStateException("block has wrong type");
        }
        ContextPage page = ((BlockWithContextPage) model).getCampaign();
        if (page == null) {
            throw new IllegalStateException("'campaign' not loaded for block");
        }
        return page;
    }

    public boolean isAdfoxBlock(BlockWithAdfox model) {
        Boolean adfoxBlock = model.getAdfoxBlock();
        if (adfoxBlock == null) {
            throw new IllegalStateException("'adfoxBlock' not loaded for block");
        }
        return adfoxBlock;
    }

    public enum BlockActionCheck implements ActionCheckId {
        IS_ADFOX_BLOCK,
        NOT_ADFOX_BLOCK,
        PAGE_NOT_PROTECTED,
        PAGE_NOT_BLOCKED,
        VALID_SITE_VERSION,
        PAGE_ALLOWS_START_BLOCK,
        PAGE_ALLOWS_RESTORE_BLOCK,
        BLOCKS_LIMIT_NOT_EXCEEDED,
        PAGE_NOT_REJECTED,
        HAS_SITE_VERSION_FEATURE
    }

}
