package ru.yandex.partner.core.entity.block.multistate;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.block.actions.BlockActionsEnum;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.messages.BlockActionMsg;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.block.BlockMultistate;
import ru.yandex.partner.core.multistate.block.BlockStateFlag;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.multistate.action.ActionCheckId;
import ru.yandex.partner.libs.multistate.action.ActionEntry;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.AbstractMultistateGraph;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.core.multistate.block.BlockStateFlag.CHECK_STATISTICS;
import static ru.yandex.partner.core.multistate.block.BlockStateFlag.DELETED;
import static ru.yandex.partner.core.multistate.block.BlockStateFlag.DELETED_WITH_PAGE;
import static ru.yandex.partner.core.multistate.block.BlockStateFlag.NEED_UPDATE;
import static ru.yandex.partner.core.multistate.block.BlockStateFlag.UPDATING;
import static ru.yandex.partner.core.multistate.block.BlockStateFlag.WORKING;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.empty;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;

@Component
public class RtbBlockMultistateGraph extends AbstractMultistateGraph<RtbBlock, BlockStateFlag> {

    private final RtbBlockActionChecksService blockActionChecksService;

    public RtbBlockMultistateGraph(RtbBlockActionChecksService blockActionChecksService) {
        super();
        this.blockActionChecksService = blockActionChecksService;
    }

    @Override
    public Set<ModelProperty<? super RtbBlock, ?>> getAllRequiredProperties() {
        return this.getAllAvailableActionNames().stream()
                .map(this::getRequiredPropertiesByActionName)
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
    }

    @Override
    public Set<ModelProperty<? super RtbBlock, ?>> getRequiredPropertiesByActionName(String actionName) {
        var actionEntry = this.getActionEntry(actionName);

        Set<ModelProperty<? super RtbBlock, ?>> requiredProperties = new HashSet<>(actionEntry.getRequiredProperties());

        var requiredPropertiesFromChecks = actionEntry.getChecks().stream()
                .map(check -> blockActionChecksService.getActionCheck(check).getRequiredModelProperties())
                .flatMap(Set::stream).collect(Collectors.toSet());

        requiredProperties.addAll(requiredPropertiesFromChecks);

        return requiredProperties;
    }

    @Override
    public BlockMultistate getMultistateFromModel(RtbBlock model) {
        return model.getMultistate();
    }

    @Override
    protected Multistate<BlockStateFlag> getMultistateForValue(Long multstateValue) {
        return new BlockMultistate(multstateValue);
    }

    @Override
    protected List<Boolean> performCheck(ActionCheckId check, List<? extends RtbBlock> models) {
        if (!(check instanceof RtbBlockActionChecksService.BlockActionCheck)) {
            return super.performCheck(check, models);
        }
        Function<List<? extends RtbBlock>, List<Boolean>> checkFunction =
                blockActionChecksService.getActionCheck(check).getCheck();
        return checkFunction == null ? super.performCheck(check, models)
                : checkFunction.apply(models);
    }

    @Override
    protected Map<ActionNameHolder, ActionEntry<RtbBlock, BlockStateFlag>> createGraph() {
        Map<ActionNameHolder, ActionEntry<RtbBlock, BlockStateFlag>> actionEntryMap = new HashMap<>();
        actionEntryMap.put(BlockActionsEnum.ADD,
                getActionEntryBuilder(BlockActionMsg.ADD, Set.of())
                        .setPredicate(empty())
                        .build());

        actionEntryMap.put(BlockActionsEnum.DELETE,
                getActionEntryBuilder(BlockActionMsg.ARCHIVE)
                        .setPredicate(not(has(DELETED)))
                        .setChecks(List.of(
                                RtbBlockActionChecksService.BlockActionCheck.NOT_ADFOX_BLOCK,
                                RtbBlockActionChecksService.BlockActionCheck.PAGE_NOT_PROTECTED
                        ))
                        .setSetFlags(Map.of(
                                DELETED, true
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.DELETE_WITH_PAGE,
                getActionEntryBuilder(BlockActionMsg.ARCHIVE_WITH_PAGE)
                        .setPredicate(not(has(DELETED)))
                        .setSetFlags(Map.of(
                                DELETED_WITH_PAGE, true,
                                DELETED, true
                        ))
                        .build());
        actionEntryMap.put(BlockActionsEnum.RESTORE,
                getActionEntryBuilder(BlockActionMsg.RESTORE)
                        .setPredicate(has(DELETED).and(not(has(DELETED_WITH_PAGE))))
                        .setChecks(List.of(
                                RtbBlockActionChecksService.BlockActionCheck.NOT_ADFOX_BLOCK,
                                RtbBlockActionChecksService.BlockActionCheck.PAGE_NOT_PROTECTED,
                                RtbBlockActionChecksService.BlockActionCheck.VALID_SITE_VERSION,
                                RtbBlockActionChecksService.BlockActionCheck.PAGE_ALLOWS_RESTORE_BLOCK,
                                RtbBlockActionChecksService.BlockActionCheck.BLOCKS_LIMIT_NOT_EXCEEDED,
                                RtbBlockActionChecksService.BlockActionCheck.HAS_SITE_VERSION_FEATURE
                        ))
                        .setSetFlags(Map.of(
                                DELETED, false
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.RESTORE_WITH_PAGE,
                getActionEntryBuilder(BlockActionMsg.RESTORE_WITH_PAGE)
                        .setPredicate(has(DELETED).and(has(DELETED_WITH_PAGE)))
                        .setChecks(List.of(
                                RtbBlockActionChecksService.BlockActionCheck.NOT_ADFOX_BLOCK,
                                RtbBlockActionChecksService.BlockActionCheck.VALID_SITE_VERSION
                        ))
                        .setSetFlags(Map.of(
                                DELETED_WITH_PAGE, false,
                                DELETED, false
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.EDIT,
                getActionEntryBuilder(BlockActionMsg.EDIT)
                        .setPredicate(not(has(DELETED)))
                        .setChecks(List.of(
                                RtbBlockActionChecksService.BlockActionCheck.PAGE_NOT_PROTECTED,
                                RtbBlockActionChecksService.BlockActionCheck.PAGE_NOT_BLOCKED,
                                RtbBlockActionChecksService.BlockActionCheck.PAGE_NOT_REJECTED
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.START,
                getActionEntryBuilder(BlockActionMsg.START)
                        .setPredicate(not(has(WORKING)).and(not(has(DELETED))))
                        .setChecks(List.of(
                                RtbBlockActionChecksService.BlockActionCheck.PAGE_ALLOWS_START_BLOCK
                        ))
                        .setSetFlags(Map.of(
                                WORKING, true
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.STOP,
                getActionEntryBuilder(BlockActionMsg.STOP)
                        .setPredicate(has(WORKING))
                        .setSetFlags(Map.of(
                                WORKING, false
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.SET_CHECK_STATISTICS,
                getActionEntryBuilder(BlockActionMsg.SET_CHECK_STATISTICS)
                        .setPredicate(not(has(DELETED)))
                        .setSetFlags(Map.of(
                                CHECK_STATISTICS, true
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.RESET_CHECK_STATISTICS,
                getActionEntryBuilder(BlockActionMsg.RESET_CHECK_STATISTICS)
                        .setPredicate(has(CHECK_STATISTICS))
                        .setSetFlags(Map.of(
                                CHECK_STATISTICS, false
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.DUPLICATE,
                getActionEntryBuilder(BlockActionMsg.DUPLICATE)
                        .setPredicate(not(has(DELETED).or(has(DELETED_WITH_PAGE))))
                        .setChecks(List.of(
                                RtbBlockActionChecksService.BlockActionCheck.NOT_ADFOX_BLOCK,
                                RtbBlockActionChecksService.BlockActionCheck.PAGE_NOT_PROTECTED,
                                RtbBlockActionChecksService.BlockActionCheck.PAGE_NOT_REJECTED,
                                RtbBlockActionChecksService.BlockActionCheck.BLOCKS_LIMIT_NOT_EXCEEDED
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.SET_NEED_UPDATE,
                getActionEntryBuilder(BlockActionMsg.SET_NEED_UPDATE)
                        .setSetFlags(Map.of(
                                NEED_UPDATE, true
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.START_UPDATE,
                getActionEntryBuilder(BlockActionMsg.START_UPDATE)
                        .setPredicate(has(NEED_UPDATE))
                        .setSetFlags(Map.of(
                                NEED_UPDATE, false,
                                UPDATING, true
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.STOP_UPDATE,
                getActionEntryBuilder(BlockActionMsg.STOP_UPDATE)
                        .setPredicate(has(UPDATING))
                        .setSetFlags(Map.of(
                                UPDATING, false
                        ))
                        .build());

        actionEntryMap.put(BlockActionsEnum.DELETE_FROM_ADFOX,
                getActionEntryBuilder(BlockActionMsg.ARCHIVE)
                        .setPredicate(not(has(DELETED)))
                        .setChecks(List.of(
                                RtbBlockActionChecksService.BlockActionCheck.IS_ADFOX_BLOCK
                        ))
                        .setSetFlags(Map.of(
                                DELETED, true
                        ))
                        .build());

        return actionEntryMap;
    }

    private ActionEntry.Builder<RtbBlock, BlockStateFlag> getActionEntryBuilder(GettextMsg titleMsg) {
        return getActionEntryBuilder(titleMsg, Set.of(RtbBlock.ID, RtbBlock.MULTISTATE));
    }

    @Override
    public Class<RtbBlock> getModelClass() {
        return RtbBlock.class;
    }

    @Override
    public Multistate<BlockStateFlag> convertMultistate(List<BlockStateFlag> enabledFlags) {
        return new BlockMultistate(enabledFlags);
    }
}
