package ru.yandex.partner.core.entity.block.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Table;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.repository.filter.ConditionFilter;
import ru.yandex.partner.core.block.BlockType;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.filter.BlockModelFilterContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.repository.filter.BlockFilterFactory;
import ru.yandex.partner.core.entity.block.repository.filter.BlockUniqueIdFilter;
import ru.yandex.partner.core.entity.block.repository.type.BlockRepositoryTypeSupportFacade;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.BinaryOperator;
import ru.yandex.partner.core.multitype.repository.PartnerTypedRepository;

import static ru.yandex.partner.core.filter.operator.FilterOperator.EQUALS;
import static ru.yandex.partner.dbschema.partner.Tables.CONTEXT_ON_SITE_RTB;

@Repository
@ParametersAreNonnullByDefault
public class BlockTypedRepository extends PartnerTypedRepository<BaseBlock, BlockType, BlockContainer,
        BlockContainer> {
    @Autowired
    public BlockTypedRepository(BlockRepositoryTypeSupportFacade typeSupportFacade,
                                BlockModelFilterContainer blockMetaFilterService,
                                DSLContext dslContext) {
        super(dslContext, typeSupportFacade, blockMetaFilterService);
    }

    @Override
    protected ConditionFilter getIdFilter(Collection<Long> blockIds) {
        return BlockFilterFactory.blockCompositeIdsFilter(blockIds);
    }

    @Override
    protected Table<?> getBaseTable() {
        return CONTEXT_ON_SITE_RTB;
    }


    @Nullable
    public BaseBlock getBlockByCompositeId(Long compositeId) {
        List<BaseBlock> typedModelByIds = getBlocksByCompositeIds(List.of(compositeId));
        if (typedModelByIds != null && !typedModelByIds.isEmpty()) {
            return typedModelByIds.get(0);
        }
        return null;
    }

    public BaseBlock getBlockByPageIdAndBlockId(Long pageId, Long blockId) {
        CoreFilterNode<BaseBlock> pageIdFilter =
                CoreFilterNode.create(BlockFilters.PAGE_ID, EQUALS, pageId);
        CoreFilterNode<BaseBlock> blockIdFilter =
                CoreFilterNode.create(BlockFilters.BLOCK_ID, EQUALS, blockId);

        List<BaseBlock> typedModelByIds = getAll(QueryOpts.forClass(BaseBlock.class)
                .withFilter(new CoreFilterNode<>(BinaryOperator.AND, List.of(pageIdFilter, blockIdFilter)))
        );

        if (typedModelByIds != null && !typedModelByIds.isEmpty()) {
            return typedModelByIds.get(0);
        }
        return null;
    }

    public List<BaseBlock> getBlocksByCompositeIds(List<Long> compositeIds) {
        var compositeIdFilter = new BlockUniqueIdFilter(compositeIds);
        return getTypedModelByIds(compositeIdFilter, null);
    }

    @Override
    protected <S extends BaseBlock> CoreFilterNode<S> getCoreFilterNodeById(Collection<Long> ids) {
        return CoreFilterNode.in(BlockFilters.ID, ids);
    }

    @Override
    public ModelProperty<? extends Model, Long> getIdModelProperty() {
        return BaseBlock.ID;
    }

}
