package ru.yandex.partner.core.entity.block.repository.type;

import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record;
import org.jooq.SelectQuery;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.block.BlockModelTypesHolder;
import ru.yandex.partner.core.block.BlockType;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.modelmeta.ModelMetaCollector;
import ru.yandex.partner.core.multitype.repository.PartnerRepositoryTypeSupport;
import ru.yandex.partner.core.multitype.repository.PartnerRepositoryTypeSupportFacade;
import ru.yandex.partner.dbschema.partner.tables.ContextOnSiteRtb;

import static ru.yandex.partner.core.entity.block.service.BlockUpdateOperationConstants.BLOCK_TYPE_SUPPORT_CLASS_WHITE_LIST;

@Component
@ParametersAreNonnullByDefault
public class BlockRepositoryTypeSupportFacade
        extends PartnerRepositoryTypeSupportFacade<BaseBlock, BlockType, BlockContainer, BlockContainer> {
    private final BlockModelTypesHolder typesHolder;

    @Autowired
    public BlockRepositoryTypeSupportFacade(
            List<PartnerRepositoryTypeSupport<? extends BaseBlock, BlockContainer, BlockContainer>>
                    typeSupports, BlockModelTypesHolder typesHolder) {
        // Сейчас выделенный тип блока отсутствует, так как читаем только из таблицы CONTEXT_ON_SITE_RTB,
        // поэтому всегда передаём в родителя конструктор RtbBlock'а
        // После объединения блоков в общую таблицу тип блока будет определяться из выделенного поля таблицы
        // (смотри закомментированный код в методе getModelType) и маппиться в enum класс типа блока BlockType.
        // Этот тип будет использоваться в map инициализаторов блоков (смотри закомменитрованный код ниже)
        super(
                typeSupports,
                ModelMetaCollector.getModelMetaHolders()
                        .stream()
                        .filter(modelMetaHolder -> BaseBlock.class.isAssignableFrom(modelMetaHolder.getEntityClass()))
                        .map(modelMetaHolder -> {
                            var name = modelMetaHolder.getName();
                            var blockType = BlockType.from(name);
                            if (blockType == null) {
                                throw new RuntimeException("Cannot find BlockType for " + name);
                            }
                            if (blockType.getConstructor() == null) {
                                throw new RuntimeException("Constructor cannot be null. Model = " + name);
                            }
                            return blockType;
                        })
                        .collect(Collectors.toMap(Function.identity(), BlockType::getConstructor)),
                Set.of(ContextOnSiteRtb.CONTEXT_ON_SITE_RTB.MODEL),
                BLOCK_TYPE_SUPPORT_CLASS_WHITE_LIST);
        this.typesHolder = typesHolder;
    }

    @Override
    protected BlockType getModelType(Record record) {
        BlockType blockType = BlockType.from(record.get(ContextOnSiteRtb.CONTEXT_ON_SITE_RTB.MODEL).getLiteral());

        if (blockType == null) {
            throw new IllegalStateException("DB returned unknown BlockType. Record: " + record);
        }

        return blockType;
    }

    @Override
    protected void addModelConditionsToQuery(SelectQuery<?> query, @Nullable Class<? extends BaseBlock> clazz) {
        if (clazz == null) {
            return;
        }

        var type = typesHolder.getTypeForDb(clazz);
        if (type != null) {
            query.addConditions(ContextOnSiteRtb.CONTEXT_ON_SITE_RTB.MODEL.eq(type));
        }
    }
}
