package ru.yandex.partner.core.entity.block.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.partner.core.action.ActionPerformer;
import ru.yandex.partner.core.action.result.ActionsResult;
import ru.yandex.partner.core.entity.IncomingFields;
import ru.yandex.partner.core.entity.block.actions.all.BlockActionAdd;
import ru.yandex.partner.core.entity.block.actions.all.factories.BlockAddActionFactory;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithMultistate;
import ru.yandex.partner.core.entity.block.service.type.add.BlockAddOperationFactory;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.entity.page.model.PageWithOwner;
import ru.yandex.partner.core.entity.page.service.ReachablePageService;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.utils.ModelPropertyUtils;
import ru.yandex.partner.core.utils.converter.MassResultConverter;

@Service
public class BlockAddServiceImpl implements BlockAddService<BlockWithMultistate> {
    private final ActionPerformer actionPerformer;
    private final BlockAddOperationFactory blockAddOperationFactory;
    private final Map<Class<BlockWithMultistate>, ReachablePageService<BlockWithMultistate>> reachablePageServiceMap;
    private final Map<Class<BlockWithMultistate>, BlockAddActionFactory<BlockWithMultistate>> factoryMap;

    @Autowired
    public BlockAddServiceImpl(ActionPerformer actionPerformer,
                               BlockAddOperationFactory blockAddOperationFactory,
                               List<ReachablePageService<? extends BlockWithMultistate>> reachablePageServices,
                               List<BlockAddActionFactory<? extends BlockWithMultistate>> factories) {
        // todo тут есть привязка к модели
        this.actionPerformer = actionPerformer;
        this.blockAddOperationFactory = blockAddOperationFactory;
        //noinspection unchecked
        this.reachablePageServiceMap = reachablePageServices.stream()
                .map(reachablePageService -> (ReachablePageService<BlockWithMultistate>) reachablePageService)
                .collect(Collectors.toMap(ReachablePageService::getBlockClass, Function.identity()));
        //noinspection unchecked
        this.factoryMap = factories.stream()
                .map(factory -> (BlockAddActionFactory<BlockWithMultistate>) factory)
                .collect(Collectors.toMap(BlockAddActionFactory::getModelClass, Function.identity()));
    }

    @Override
    public <B extends BlockWithMultistate, P extends PageWithOwner> ActionsResult<?> addModels(
            Class<B> blockClass,
            List<B> blocks,
            Class<P> pageClass,
            IncomingFields incomingFields,
            OperationMode mode,
            Consumer<BlockContainer> configurer
    ) {
        var blockContainer = blockAddOperationFactory
                // todo ContextPage зашит в BlockContainer
                .prepareAddOperationContainer(mode, incomingFields);
        configurer.accept(blockContainer);

        Map<Long, P> reachablePages = getReachablePages(blockClass, blocks, pageClass,
                (CoreFilterNode<? super P>) blockContainer.getPageReachabilityFilter(blockClass));
        blockContainer.getReachablePages().put(blockClass, reachablePages);

        MassResult<Long> massResult = blockAddOperationFactory
                .createAddOperation(Applicability.PARTIAL, blocks, blockContainer)
                .prepareAndApply();

        Map<Long, BlockWithMultistate> addedModels = Maps.newHashMapWithExpectedSize(massResult.getResult().size());
        Map<Long, Set<ModelProperty<? super BlockWithMultistate, ?>>> addedProps =
                Maps.newHashMapWithExpectedSize(massResult.getResult().size());

        for (int i = 0; i < massResult.getResult().size(); i++) {
            addedModels.put(massResult.getResult().get(i).getResult(), blocks.get(i));
            Collection<ModelProperty<?, ?>> fields = incomingFields.getUpdatedFields(blocks.get(i));
            Set<ModelProperty<? super BlockWithMultistate, ?>> fieldsSet = fields == null
                    ? Set.of()
                    : ModelPropertyUtils.castSuper(fields);

            addedProps.put(blocks.get(i).getId(), fieldsSet);
        }

        ActionsResult<?> actionResult;
        if (massResult.isSuccessful() && massResult.getErrorCount() == 0) {
            var actionAdd = getAction(blockClass, addedModels, addedProps);
            actionResult = actionPerformer.doActions(actionAdd);
        } else {
            actionResult = new ActionsResult<>();
            actionResult.setCommitted(false);
        }

        actionResult.putToResults(blockClass, MassResultConverter.convertMassResult(massResult));
        return actionResult;
    }

    private BlockActionAdd<BlockWithMultistate> getAction(
            Class<? extends BlockWithMultistate> blockClass,
            Map<Long, BlockWithMultistate> addedModels,
            Map<Long, Set<ModelProperty<? super BlockWithMultistate, ?>>> addedProps) {
        return Objects.requireNonNull(factoryMap.get(blockClass),
                        "Unsupported class for Action Add. Class: " + blockClass)
                .createAction(addedModels, addedProps);
    }

    @Override
    public <B extends BlockWithMultistate, P extends BasePage> Map<Long, P> getReachablePages(
            Class<B> blockClass,
            List<B> blocks,
            Class<P> pageClass,
            CoreFilterNode<? super P> reachablePagesFilter) {
        var reachablePageService = reachablePageServiceMap.get(blockClass);

        if (reachablePageService == null) {
            return Map.of();
        }

        var pageIds = blocks.stream()
                .map(BaseBlock::getPageId)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        if (!pageIds.isEmpty()) {
            var pageFilter = CoreFilterNode.<P>and(
                    reachablePagesFilter,
                    reachablePageService.getPageIdMetaFilter().in(pageIds)
            );
            return reachablePageService.getReachablePagesForAdd(pageFilter, pageClass)
                    .stream()
                    .collect(Collectors.toMap(BasePage::getId, Function.identity()));
        } else {
            return Map.of();
        }

    }
}
