package ru.yandex.partner.core.entity.block.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.typesupport.TypeFilteringUtils;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ChangesAppliedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.ModelChangesValidatedStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.block.BlockUniqueIdConverter;
import ru.yandex.partner.core.entity.IncomingFields;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.container.BlockContainerImpl;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithDesignTemplates;
import ru.yandex.partner.core.entity.block.repository.BlockModifyRepository;
import ru.yandex.partner.core.entity.block.repository.BlockTypedRepository;
import ru.yandex.partner.core.entity.block.service.type.update.BlockUpdateOperationTypeSupportFacade;
import ru.yandex.partner.core.entity.block.service.validation.type.BlockValidationTypeSupportFacade;
import ru.yandex.partner.core.multitype.service.validation.type.update.EditableFieldValidator;
import ru.yandex.partner.core.validation.HasValidationContainer;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class BlockUpdateOperation extends SimpleAbstractUpdateOperation<BaseBlock, Long> implements
        HasValidationContainer<BaseBlock, BlockContainer> {
    private final DSLContext dslContext;

    private final BlockContainer operationContainer;

    private final BlockTypedRepository typedRepository;
    private final BlockModifyRepository modifyRepository;
    private final BlockUpdateOperationTypeSupportFacade updateOperationTypeSupportFacade;
    private final BlockValidationTypeSupportFacade validationTypeSupportFacade;
    private final Collection<BaseBlock> preloadedModels;
    private final EditableFieldValidator<BaseBlock> editableFieldValidator;
    private final List<ModelChanges<BaseBlock>> modelChanges;

    @SuppressWarnings("checkstyle:parameternumber")
    public BlockUpdateOperation(Applicability applicability,
                                List<ModelChanges<BaseBlock>> modelChanges,
                                IncomingFields incomingFields,
                                DSLContext dslContext,
                                BlockTypedRepository typedRepository,
                                BlockModifyRepository modifyRepository,
                                BlockUpdateOperationTypeSupportFacade updateOperationTypeSupportFacade,
                                BlockValidationTypeSupportFacade validationTypeSupportFacade,
                                @Nullable Collection<BaseBlock> preloadedModels,
                                EditableFieldValidator<BaseBlock> editableFieldValidator,
                                Function<Long, BaseBlock> modelStubCreator) {
        super(applicability, modelChanges, modelStubCreator);
        this.dslContext = dslContext;
        this.typedRepository = typedRepository;
        this.modifyRepository = modifyRepository;
        this.updateOperationTypeSupportFacade = updateOperationTypeSupportFacade;
        this.validationTypeSupportFacade = validationTypeSupportFacade;
        this.editableFieldValidator = editableFieldValidator;
        this.operationContainer = BlockContainerImpl.create(OperationMode.EDIT);
        this.operationContainer.setIncomingFields(incomingFields);
        this.preloadedModels = preloadedModels;
        this.modelChanges = modelChanges;
    }

    @Override
    protected Collection<BaseBlock> getModels(Collection<Long> ids) {
        if (ids.isEmpty()) {
            return List.of();
        }

        List<BaseBlock> baseBlocks;
        if (preloadedModels != null) {
            // todo PI-27940 корректно вычислять
            Class<? extends BaseBlock> clazz = ids.stream().findAny().map(BlockUniqueIdConverter.Prefixes::byUniqueId)
                    .map(prefixes -> (Class<BaseBlock>) prefixes.getModelClass()).orElse(BaseBlock.class);

            Set<ModelProperty<? extends Model, ?>> affectedProps = updateOperationTypeSupportFacade
                    .getAffectedProps(modelChanges, clazz);

            boolean isSomeModelsMissProps = preloadedModels.stream()
                    .flatMap(model -> affectedProps.stream().map(prop -> ((ModelProperty<Model, ?>) prop).get(model)))
                    .anyMatch(Objects::isNull);

            if (isSomeModelsMissProps) { // если UpdateOperation был создан вне action edit
                baseBlocks = typedRepository.getTyped(ids, true);
            } else {
                baseBlocks = preloadedModels.stream()
                        .filter(model -> ids.contains(model.getId()))
                        .collect(Collectors.toList());
            }
        } else {
            // Здесь сейчас нет возможности считать не все так как
            baseBlocks = typedRepository.getTyped(ids, true);
        }

        return baseBlocks;
    }

    @Override
    protected ValidationResult<List<ModelChanges<BaseBlock>>, Defect> validateModelChanges(
            List<ModelChanges<BaseBlock>> modelChanges) {
        ValidationResult<List<ModelChanges<BaseBlock>>, Defect> vr =
                new ValidationResult<>(modelChanges);
        validationTypeSupportFacade.updateValidateModelChanges(operationContainer, vr);
        return vr;
    }

    @Override
    protected void onModelChangesValidated(ModelChangesValidatedStep<BaseBlock> modelChangesValidatedStep) {
        updateOperationTypeSupportFacade.onModelChangesValidated(operationContainer,
                modelChangesValidatedStep.getValidModelChanges()
        );
    }

    @Override
    protected ValidationResult<List<ModelChanges<BaseBlock>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<BaseBlock>>, Defect> preValidateResult, Map<Long, BaseBlock> models) {
        editableFieldValidator.updateValidateBeforeApply(preValidateResult, models);
        //добавляем неизмененные дизайны в контейнер на этом шаге, чтобы потом их не вычитывать
        List<BlockWithDesignTemplates> blocks = TypeFilteringUtils
                .filterModelsOfClass(models.values(), BlockWithDesignTemplates.class);
        operationContainer.setUnmodifiedTemplates(StreamEx.of(blocks).filter(it -> it.getDesignTemplates() != null)
                .collect(Collectors.toMap(BaseBlock::getId, BlockWithDesignTemplates::getDesignTemplates)));

        validationTypeSupportFacade.updateValidateBeforeApply(operationContainer, preValidateResult, models);
        return preValidateResult;
    }

    @Override
    protected void onChangesApplied(ChangesAppliedStep<BaseBlock> changesAppliedStep) {
        List<AppliedChanges<BaseBlock>> validAppliedChanges =
                new ArrayList<>(changesAppliedStep.getAppliedChangesForValidModelChanges());
        updateOperationTypeSupportFacade.onChangesApplied(operationContainer, validAppliedChanges);
    }

    /**
     * В этом вызове проверяются права пользователя на редактирование отдельных полей.
     * Вынесена в этот метод, так как необходимы предыдущие значения всех полей, для правильного сравнения вложенных
     *
     * @param validationResult                   - результат валидации
     * @param appliedChangesForValidModelChanges - AppliedChanges для валидных моделей
     * @return - обновлённый результат валидации
     */
    @Override
    protected ValidationResult<List<BaseBlock>, Defect> validateAppliedChanges(
            ValidationResult<List<BaseBlock>, Defect> validationResult,
            Map<Integer, AppliedChanges<BaseBlock>> appliedChangesForValidModelChanges) {

        validationTypeSupportFacade.updateValidateAppliedChanges(operationContainer, validationResult,
                appliedChangesForValidModelChanges);
        if (!validationResult.hasAnyErrors()) {
            validationTypeSupportFacade.validate(operationContainer, validationResult,
                    appliedChangesForValidModelChanges);
        }
        return validationResult;
    }

    @Override
    protected void onAppliedChangesValidated(AppliedChangesValidatedStep<BaseBlock> appliedChangesValidatedStep) {
        var validAppliedChanges = new ArrayList<>(appliedChangesValidatedStep.getValidAppliedChanges());

        updateOperationTypeSupportFacade.onAppliedChangesValidated(operationContainer, validAppliedChanges);
    }

    @Override
    protected void beforeExecution(ExecutionStep<BaseBlock> executionStep) {
        var appliedChangesForExecution = new ArrayList<>(executionStep.getAppliedChangesForExecution());

        updateOperationTypeSupportFacade.beforeExecution(appliedChangesForExecution, operationContainer);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<BaseBlock>> applicableAppliedChanges) {
        updateOperationTypeSupportFacade.addToAdditionalActionsContainer(operationContainer,
                operationContainer,
                applicableAppliedChanges);
        dslContext.transaction(configuration -> {
            processUpdateTask(configuration.dsl(), applicableAppliedChanges);
            updateOperationTypeSupportFacade.updateRelatedEntitiesInTransaction(
                    configuration.dsl(),
                    operationContainer,
                    applicableAppliedChanges
            );
        });
        return mapList(applicableAppliedChanges, b -> b.getModel().getId());
    }

    private void processUpdateTask(DSLContext dsl, List<AppliedChanges<BaseBlock>> validAppliedChanges) {
        modifyRepository.update(dsl, operationContainer, validAppliedChanges);
    }

    @Override
    protected void afterExecution(ExecutionStep<BaseBlock> executionStep) {
        var appliedChangesForExecution = new ArrayList<>(executionStep.getAppliedChangesForExecution());
        updateOperationTypeSupportFacade.afterExecution(appliedChangesForExecution, operationContainer);
    }

    @Override
    public BlockContainer getContainer() {
        return operationContainer;
    }
}
