package ru.yandex.partner.core.entity.block.type.base;

import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import NPartner.Page.TPartnerPage;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.partner.core.block.BlockModelTypesHolder;
import ru.yandex.partner.core.entity.block.container.BlockBkDictContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.repository.BlockBkFiller;
import ru.yandex.partner.core.entity.block.repository.type.AbstractBlockRepositoryTypeSupportWithMapper;
import ru.yandex.partner.core.holder.ModelPropertiesHolder;
import ru.yandex.partner.dbschema.partner.enums.ContextOnSiteRtbModel;
import ru.yandex.partner.dbschema.partner.tables.ContextOnSiteRtb;

import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.partner.core.entity.block.model.prop.BaseBlockBlockIdPropHolder.BLOCK_ID;
import static ru.yandex.partner.core.entity.block.model.prop.BaseBlockIdPropHolder.ID;
import static ru.yandex.partner.core.entity.block.model.prop.BaseBlockPageIdPropHolder.PAGE_ID;
import static ru.yandex.partner.core.entity.block.type.base.BaseBlockConstants.EDIT_FORBIDDEN_MODEL_PROPERTIES;
import static ru.yandex.partner.core.holder.ModelPropertiesHolder.fromModelProperties;
import static ru.yandex.partner.dbschema.partner.Tables.CONTEXT_ON_SITE_RTB;

@Component
@ParametersAreNonnullByDefault
public class BaseBlockRepositoryTypeSupport
        extends AbstractBlockRepositoryTypeSupportWithMapper<BaseBlock>
        implements BlockBkFiller<BaseBlock> {

    private final JooqMapper<BaseBlock> mapper;
    private final BlockModelTypesHolder typesHolder;

    @Autowired
    protected BaseBlockRepositoryTypeSupport(DSLContext dslContext, BlockModelTypesHolder typesHolder) {
        super(dslContext);
        this.typesHolder = typesHolder;
        this.mapper = createCommonBlockMapper();
    }

    private static JooqMapper<BaseBlock> createCommonBlockMapper() {
        return JooqMapperBuilder.<BaseBlock>builder()
                .map(property(ID, CONTEXT_ON_SITE_RTB.UNIQUE_ID))
                .map(property(PAGE_ID, CONTEXT_ON_SITE_RTB.CAMPAIGN_ID))
                .map(property(BLOCK_ID, CONTEXT_ON_SITE_RTB.ID))
                .build();
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, BaseBlock block) {
        var type = typesHolder.getTypeForDb(block.getClass());
        if (type == null) {
            throw new RuntimeException("Can't create block with unsupported type");
        }

        insertHelperAggregator.getOrCreate(ContextOnSiteRtb.CONTEXT_ON_SITE_RTB)
                .add(getJooqMapper(), block)
                .set(ContextOnSiteRtb.CONTEXT_ON_SITE_RTB.MODEL, type);
    }

    @Override
    public JooqMapper<BaseBlock> getJooqMapper() {
        return mapper;
    }

    @Override
    public Class<BaseBlock> getTypeClass() {
        return BaseBlock.class;
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BaseBlock> blocks) {
    }

    @Override
    public ModelPropertiesHolder getEditableModelProperties(BaseBlock model) {
        return fromModelProperties(
                new HashSet<>(
                        Sets.difference(getJooqMapper().getWritableModelProperties(), EDIT_FORBIDDEN_MODEL_PROPERTIES)
                )
        );
    }

    @Override
    public void fillBkData(@NotNull BaseBlock block, @NotNull TPartnerPage.TBlock.Builder bkData,
                           @NotNull BlockBkDictContainer container) {
        bkData.setBlockID(block.getBlockId());
        ContextOnSiteRtbModel model = Objects.requireNonNull(typesHolder.getTypeForDb(block.getClass()),
                "Cannot find out ContextOnSiteRtbModel for (Class, Id): " + block.getClass() + ", " + block.getId());
        bkData.setBlockModel(model.getLiteral());
    }
}
