package ru.yandex.partner.core.entity.block.type.brands;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import NPartner.Page.TPartnerPage;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.PageBlockIds;
import ru.yandex.partner.core.entity.block.container.BlockBkDictContainer;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithBrands;
import ru.yandex.partner.core.entity.block.model.Brand;
import ru.yandex.partner.core.entity.block.repository.BlockBkFiller;
import ru.yandex.partner.core.entity.block.repository.type.AbstractBlockRepositoryTypeSupport;
import ru.yandex.partner.core.entity.block.repository.type.BlockRepositoryTypeSupportWithoutMapper;
import ru.yandex.partner.core.entity.brand.BrandRepository;
import ru.yandex.partner.core.entity.brand.filter.BrandFilters;
import ru.yandex.partner.core.entity.common.utils.ModelWithBlockPageIdsUtils;
import ru.yandex.partner.core.holder.ModelPropertiesHolder;
import ru.yandex.partner.core.multitype.repository.relation.RelationRepository;
import ru.yandex.partner.core.utils.FunctionalUtils;

import static ru.yandex.partner.core.holder.ModelPropertiesHolder.fromModelProperties;

@Component
@ParametersAreNonnullByDefault
public class BlockWithBrandsRepositoryTypeSupport extends AbstractBlockRepositoryTypeSupport<BlockWithBrands>
        implements BlockRepositoryTypeSupportWithoutMapper<BlockWithBrands>, BlockBkFiller<BlockWithBrands> {
    private final BrandRepository brandRepository;
    private final Set<ModelProperty<? super BlockWithBrands, ?>> affectedModelProperties;
    private final Set<ModelProperty<? super BlockWithBrands, ?>> editableModelProperties;

    @Autowired
    public BlockWithBrandsRepositoryTypeSupport(DSLContext dslContext, BrandRepository brandRepository) {
        super(dslContext);
        this.brandRepository = brandRepository;
        this.affectedModelProperties = Set.of(BlockWithBrands.BRANDS, BlockWithBrands.HAS_BRANDS_SETTINGS);
        this.editableModelProperties = Set.of(BlockWithBrands.BRANDS);
    }

    @Override
    public Class<BlockWithBrands> getTypeClass() {
        return BlockWithBrands.class;
    }

    @Override
    public Set<ModelProperty<? super BlockWithBrands, ?>> getAffectedModelProperties() {
        return affectedModelProperties;
    }

    @Override
    public ModelPropertiesHolder getEditableModelProperties(BlockWithBrands model) {
        return fromModelProperties(new HashSet<>(editableModelProperties));
    }

    @Override
    public void insertToAdditionTables(DSLContext context, BlockContainer addModelParametersContainer,
                                       Collection<BlockWithBrands> models) {
        // init backward references
        models.forEach(it ->
                ModelWithBlockPageIdsUtils.setPageBlockIds(it.getBrands(), it, addModelParametersContainer.getMode()));

        brandRepository.insert(
                StreamEx.of(models)
                        .filter(model -> model.getBrands() != null)
                        .toFlatList(BlockWithBrands::getBrands));
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BlockWithBrands> models) {
        Map<PageBlockIds, List<Brand>> result = RelationRepository.of(
                        brandRepository, BrandFilters.PAGE_ID, BrandFilters.BLOCK_ID)
                .select(dslContext,
                FunctionalUtils.mapList(models, block -> new PageBlockIds(block.getPageId(), block.getBlockId()))
        );

        for (BlockWithBrands block : models) {
            List<Brand> brands = result.get(new PageBlockIds(block.getPageId(), block.getBlockId()));
            block.setBrands(brands);
            block.setHasBrandsSettings(brands != null && !brands.isEmpty());
        }
    }

    @Override
    public void fillBkData(@NotNull BlockWithBrands model, @NotNull TPartnerPage.TBlock.Builder bkData,
                           @NotNull BlockBkDictContainer container) {
        if (model.getBrands() == null) {
            return;
        }

        for (Brand brand : model.getBrands()) {
            long value = brand.getBlocked() ? CoreConstants.BLOCKED_CPM_VALUE.longValue() :
                    brand.getCpm().multiply(BigDecimal.valueOf(1000L)).longValue();
            bkData.addBrandBuilder()
                    .setBrandID(brand.getBid())
                    .setCurrency(CoreConstants.Currency.DEFAULT)
                    .setValue(value);
        }
    }
}
