package ru.yandex.partner.core.entity.block.type.brands;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithBrands;
import ru.yandex.partner.core.entity.block.model.Brand;
import ru.yandex.partner.core.entity.block.repository.type.BlockRepositoryTypeSupportFacade;
import ru.yandex.partner.core.entity.block.service.type.update.AbstractBlockUpdateOperationTypeSupport;
import ru.yandex.partner.core.entity.brand.BrandRepository;
import ru.yandex.partner.core.entity.brand.filter.BrandFilters;
import ru.yandex.partner.core.multitype.repository.relation.CollectionRelation;
import ru.yandex.partner.core.multitype.repository.relation.Relation;
import ru.yandex.partner.core.multitype.repository.relation.RelationRepository;
import ru.yandex.partner.core.multitype.repository.relation.UpdateStrategy;

@Component
public class BlockWithBrandsUpdateOperationTypeSupport
        extends AbstractBlockUpdateOperationTypeSupport<BlockWithBrands> {

    private static final Set<ModelProperty<? super BlockWithBrands, ?>> NEED_UPDATE_IN_BK_FIELDS =
            Set.of(BlockWithBrands.BRANDS);
    private final Relation<BlockWithBrands, Brand, List<Brand>> brandsModelRelation;

    @Autowired
    public BlockWithBrandsUpdateOperationTypeSupport(BrandRepository brandRepository,
                                                     BlockRepositoryTypeSupportFacade repositoryTypeSupportFacade) {
        super(repositoryTypeSupportFacade);
        this.brandsModelRelation = new CollectionRelation<>(BlockWithBrands.BRANDS,
                RelationRepository.of(brandRepository, BrandFilters.PAGE_ID, BrandFilters.BLOCK_ID),
                UpdateStrategy.rewrite(),
                Brand::getBid);
    }

    @Override
    public Class<BlockWithBrands> getTypeClass() {
        return BlockWithBrands.class;
    }

    @Override
    public void updateRelatedEntitiesInTransaction(DSLContext dslContext, BlockContainer updateContainer,
                                                   List<AppliedChanges<BlockWithBrands>> appliedChanges) {
        fixBackReferenceForChanges(appliedChanges);
        brandsModelRelation.processUpdate(
                appliedChanges.stream()
                        .filter(changes -> changes.changed(brandsModelRelation.getProperty()))
                        .map(changes -> Pair.of(
                                changes.getOldValue(brandsModelRelation.getProperty()),
                                changes.getNewValue(brandsModelRelation.getProperty())
                        )).collect(Collectors.toList()),
                updateContainer.getIncomingFields()
        );
    }

    private void fixBackReferenceForChanges(Collection<AppliedChanges<BlockWithBrands>> appliedChanges) {
        for (var blockChanges : appliedChanges) {
            List<Brand> newBrands = blockChanges.getNewValue(BlockWithBrands.BRANDS);

            if (newBrands == null) {
                continue;
            }

            var blockId = blockChanges.getModel().getBlockId();
            var pageId = blockChanges.getModel().getPageId();

            for (Brand brand : newBrands) {
                if (brand.getBlockId() == null) {
                    brand.setBlockId(blockId);
                }

                if (brand.getPageId() == null) {
                    brand.setPageId(pageId);
                }
            }
        }
    }

    @Override
    public Set<ModelProperty<? super BlockWithBrands, ?>> needBsResyncProps() {
        return NEED_UPDATE_IN_BK_FIELDS;
    }
}
