package ru.yandex.partner.core.entity.block.type.brands;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithBrands;
import ru.yandex.partner.core.entity.block.model.Brand;
import ru.yandex.partner.core.entity.block.service.validation.type.AbstractBlockValidationTypeSupport;
import ru.yandex.partner.core.entity.brand.TnsDictBrandRepository;

@Component
public class BlockWithBrandsValidationTypeSupport extends AbstractBlockValidationTypeSupport<BlockWithBrands> {

    private final BlockWithBrandsValidatorProvider validatorProvider;

    private final DSLContext dslContext;
    private final TnsDictBrandRepository tnsDictBrandRepository;

    @Autowired
    public BlockWithBrandsValidationTypeSupport(BlockWithBrandsValidatorProvider validatorProvider,
                                                DSLContext dslContext, TnsDictBrandRepository tnsDictBrandRepository) {
        this.validatorProvider = validatorProvider;
        this.dslContext = dslContext;
        this.tnsDictBrandRepository = tnsDictBrandRepository;
    }

    @Override
    public void fillContainer(BlockContainer container, List<BlockWithBrands> blocks) {
        if (container.getAvailableBrandIds().isEmpty()) {
            container.setAvailableBrandIds(tnsDictBrandRepository.getExistingIds(
                    dslContext,
                    blocks.stream().filter(b -> b.getBrands() != null)
                            .flatMap(block -> block.getBrands().stream().filter(Objects::nonNull).map(Brand::getBid))
                            .collect(Collectors.toSet())
            ));
        }
    }

    @Override
    public void fillContainerFullDictionaries(BlockContainer container) {
        container.setAvailableBrandIds(tnsDictBrandRepository.getAllExistingIds(dslContext));
    }

    @Override
    public ValidationResult<List<BlockWithBrands>, Defect> addPreValidate(
            BlockContainer container,
            ValidationResult<List<BlockWithBrands>, Defect> vr) {
        return new ListValidationBuilder<>(vr)
                .checkEachBy(validatorProvider.addPreValidator(), When.isTrue(container.getMode().creation()))
                .getResult();
    }

    @Override
    public Class<BlockWithBrands> getTypeClass() {
        return BlockWithBrands.class;
    }

    @Override
    public ValidationResult<List<BlockWithBrands>, Defect> validate(
            BlockContainer container,
            ValidationResult<List<BlockWithBrands>, Defect> vr) {

        return new ListValidationBuilder<>(vr)
                .checkBy(validatorProvider.validator(container.getAvailableBrandIds()))
                .getResult();
    }
}
