package ru.yandex.partner.core.entity.block.type.brands;

import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.partner.core.entity.block.model.Brand;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefects;
import ru.yandex.partner.core.entity.block.service.validation.defects.StrategyDefectParams;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.greaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.partner.core.CoreConstants.MAX_CPM;
import static ru.yandex.partner.core.CoreConstants.MIN_CPM;
import static ru.yandex.partner.core.action.exception.DefectInfoWithMsgId.defect;
import static ru.yandex.partner.core.validation.defects.presentation.CommonValidationMsg.MUST_BE_UINT;

public class BrandValidator implements Validator<Brand, Defect> {

    @Override
    public ValidationResult<Brand, Defect> apply(Brand brand) {
        ModelItemValidationBuilder<Brand> vb = ModelItemValidationBuilder.of(brand);
        vb.item(Brand.BID).check(notNull()).check(greaterThan(-1L), defect(MUST_BE_UINT), When.notNull());
        vb.item(Brand.BLOCKED).check(notNull());
        vb.checkByFunction(BrandValidator::validateState, When.isValid());

        var defect = new Defect<>(BlockDefectIds.Strategy.INCORRECT_MINCPM_VALUE,
                new StrategyDefectParams().withMinCpm(MIN_CPM).withMaxCpm(MAX_CPM));
        vb.item(Brand.CPM).check(inRange(MIN_CPM, MAX_CPM), defect, When.notNull());

        return vb.getResult();
    }

    private static Defect validateState(Brand brand) {
        if (brand.getBlocked() && brand.getCpm() != null) {
            return BlockDefects.brandCpmSetForBlocked();
        } else if (!brand.getBlocked() && brand.getCpm() == null) {
            return BlockDefects.brandCpmNotSetForNonBlocked();
        }
        return null;
    }
}
