package ru.yandex.partner.core.entity.block.type.common;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithCommonFields;
import ru.yandex.partner.core.entity.dsp.model.Dsp;
import ru.yandex.partner.core.entity.dsp.multistate.DspMultistateGraph;
import ru.yandex.partner.core.entity.dsp.rules.DspRule;
import ru.yandex.partner.core.entity.dsp.rules.DspRuleComponent;
import ru.yandex.partner.core.entity.dsp.rules.DspRuleContainer;
import ru.yandex.partner.core.entity.dsp.rules.DspRuleSet;
import ru.yandex.partner.core.entity.dsp.rules.result.DspRuleResult;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.dsp.DspStateFlag;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.core.CoreConstants.DSP_OWN_ADV_ID;
import static ru.yandex.partner.core.CoreConstants.DSP_UNSOLD_ID;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.CREATED_IN_BK;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.CREATED_IN_PI;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.DELETED;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.LINKED_IN_BALANCE;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.NOT_NEED_CREATE_IN_PI;
import static ru.yandex.partner.dbschema.partner.Tables.DSP;
import static ru.yandex.partner.dbschema.partner.Tables.DSP_TYPE;


public abstract class BlockWithCommonFieldsDspRules extends DspRuleComponent<BlockWithCommonFields> {
    private final Predicate<Multistate<DspStateFlag>> saneMultistateTestCondition;
    private final List<Long> saneMultistates;
    private final Optional<List<Long>> workingOnAllPlatformsMultistates;

    @Autowired
    public BlockWithCommonFieldsDspRules(DspMultistateGraph multistateGraph) {
        this.saneMultistateTestCondition = CREATED_IN_BK.and(CREATED_IN_PI.or(NOT_NEED_CREATE_IN_PI))
                .and(LINKED_IN_BALANCE).and(not(DELETED));
        this.saneMultistates = multistateGraph.getMultistatesForPredicate(saneMultistateTestCondition)
                .stream()
                .map(Multistate::toMultistateValue)
                .sorted()
                .distinct()
                .toList();
        this.workingOnAllPlatformsMultistates = getWorkingOnAllPlatformsFlag()
                .map(stateFlag -> multistateGraph.getMultistatesForPredicate(stateFlag)
                        .stream()
                        .map(Multistate::toMultistateValue)
                        .sorted()
                        .distinct()
                        .toList()
                );

    }

    public abstract Optional<DspStateFlag> getWorkingOnAllPlatformsFlag();

    @Override
    public Class<BlockWithCommonFields> getEntireTypeClass() {
        return null;
    }

    @Override
    public DspRuleSet getDspRuleSet(DspRuleContainer container) {
        var ruleSet = new DspRuleSet()
                .withRule(
                        // dsp_type
                        new DspRule<BasePage, BlockWithCommonFields, Dsp>() {
                            @Override
                            public DspRuleResult availableDsps(
                                    DspRuleContainer dspRuleContainer, BlockWithCommonFields block) {
                                return DspRuleResult.limit(DSP_TYPE.TYPE_ID.in(dspRuleContainer.getDspTypes()));
                            }

                            @Override
                            public DspRuleResult turnOnDsps(
                                    DspRuleContainer dspRuleContainer,
                                    BasePage page, BlockWithCommonFields block, Dsp dsp) {
                                Set<Long> dspType = dspRuleContainer.getDspTypes();
                                if (dspType != null && dsp.getTypes() != null &&
                                        dsp.getTypes().stream().anyMatch(dspType::contains)
                                ) {
                                    return DspRuleResult.extendIdentity();
                                }
                                return DspRuleResult.extendZero();
                            }
                        }
                )
                .withRule(
                        // sane multistate
                        new DspRule<BasePage, BlockWithCommonFields, Dsp>() {
                            @Override
                            public DspRuleResult availableDsps(
                                    DspRuleContainer dspRuleContainer, BlockWithCommonFields block) {
                                return DspRuleResult.limit(DSP.ID.notIn(List.of(DSP_OWN_ADV_ID, DSP_UNSOLD_ID))
                                        .and(DSP.MULTISTATE.in(saneMultistates)));
                            }

                            @Override
                            public DspRuleResult turnOnDsps(
                                    DspRuleContainer dspRuleContainer, BasePage page,
                                    BlockWithCommonFields block, Dsp dsp) {
                                if (dsp.getId() != DSP_OWN_ADV_ID && dsp.getId() != DSP_UNSOLD_ID
                                        && saneMultistateTestCondition != null
                                        && saneMultistateTestCondition.test(dsp.getMultistate())) {
                                    return DspRuleResult.limitIdentity();
                                }
                                return DspRuleResult.limitZero();
                            }
                        }
                );

        workingOnAllPlatformsMultistates.ifPresent(states ->
                ruleSet.withRule(new DspRule<BasePage, BaseBlock, Dsp>() {
                    @Override
                    public DspRuleResult defaultDsps(
                            DspRuleContainer dspRuleContainer, BasePage page, BaseBlock block) {
                        return DspRuleResult.extend(DSP.MULTISTATE.in(states));
                    }

                    @Override
                    public DspRuleResult turnOnDsps(DspRuleContainer dspRuleContainer, BasePage page,
                                                    BaseBlock block, Dsp dsp) {
                        if (dsp.getMultistate().test(getWorkingOnAllPlatformsFlag().get())) {
                            return DspRuleResult.extendZero();
                        } else {
                            return DspRuleResult.extendIdentity();
                        }
                    }
                }));

        return ruleSet;
    }
}
