package ru.yandex.partner.core.entity.block.type.custombkdata;

import java.util.List;
import java.util.Objects;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.util.DefaultPrettyPrinter;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelChanges;
import ru.yandex.partner.core.block.BlockUniqueIdConverter;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithCustomBkData;
import ru.yandex.partner.core.entity.block.model.BlockWithCustomBkDataAndDesignTemplates;
import ru.yandex.partner.core.entity.block.repository.BlockModifyRepository;
import ru.yandex.partner.core.entity.block.service.OperationMode;
import ru.yandex.partner.core.entity.block.service.type.add.AbstractBlockAddOperationTypeSupport;
import ru.yandex.partner.core.entity.designtemplates.model.DesignTemplates;


@Component
public class BlockWithCustomBkDataAndDesignTemplatesAddOperationTypeSupport
        extends AbstractBlockAddOperationTypeSupport<BlockWithCustomBkDataAndDesignTemplates> {

    private final CustomBkDataService customBkDataService;
    private final BlockModifyRepository blockModifyRepository;
    private final ObjectMapper objectMapper;

    @Autowired
    public BlockWithCustomBkDataAndDesignTemplatesAddOperationTypeSupport(
            CustomBkDataService customBkDataService,
            BlockModifyRepository blockModifyRepository,
            ObjectMapper objectMapper
    ) {
        this.customBkDataService = customBkDataService;
        this.blockModifyRepository = blockModifyRepository;
        this.objectMapper = objectMapper;
    }

    @Override
    public Class<BlockWithCustomBkDataAndDesignTemplates> getTypeClass() {
        return BlockWithCustomBkDataAndDesignTemplates.class;
    }

    @Override
    public void beforeExecution(BlockContainer addContainer, List<BlockWithCustomBkDataAndDesignTemplates> models) {
        if (addContainer.getMode().equals(OperationMode.DUPLICATE)) {
            for (BlockWithCustomBkData model : models) {
                var bkDataJson = model.getBkData();
                if (bkDataJson == null) {
                    continue;
                }

                var rtbDesign = customBkDataService
                        .getDataFromBk(bkDataJson, customBkDataService.getDesignField(model.getClass()));
                if (rtbDesign.isMissingNode() || rtbDesign.isNull()) {
                    continue;
                }
                String newPublicId = BlockUniqueIdConverter.convertToPublicId(model.getId());

                // Пока возможно 2 случая. Первый когда в RtbDesign приходит сразу хэш.
                // Второй - Когда приходит список хэшей
                // TODO: когда все перейдут на МСФ должен остаться только один
                if (rtbDesign.isObject()) {
                    rtbDesign.iterator().forEachRemaining((design) -> {
                        var blockId = design.at("/design/blockId");
                        if (!blockId.isMissingNode() && !blockId.isNull()) {
                            model.setBkData(Pattern.compile("(\"blockId\"\\s*:\\s*\")" + blockId.asText())
                                    .matcher(bkDataJson)
                                    .replaceAll("\"blockId\" : \"" + newPublicId));
                        }
                    });
                } else if (rtbDesign.isTextual()) {
                    var blockId = customBkDataService.getDataFromBk("{" + rtbDesign.asText() + "}", "blockId");
                    if (!blockId.isMissingNode() && !blockId.isNull()) {
                        model.setBkData(Pattern
                                .compile("(\\\\\"blockId\\\\\"\\s*:\\s*\\\\\")" + blockId.asText() + "\\\\\"")
                                .matcher(bkDataJson)
                                .replaceAll("\\\\\"blockId\\\\\":\\\\\"" + newPublicId + "\\\\\""));
                    }
                } else {
                    throw new BkDataException("Unknown bk_data format: " + bkDataJson);
                }
            }
        }

    }

    @Override
    public void afterExecution(BlockContainer addContainer, List<BlockWithCustomBkDataAndDesignTemplates> models) {
        if (addContainer.getMode() == OperationMode.DUPLICATE) {
            var appliedBkDataChanges = models.stream()
                    .filter(block -> Objects.nonNull(block.getBkData()))
                    .map(block -> {
                        try {
                            var bkDataJson = block.getBkData();

                            var bkData = objectMapper.readTree(bkDataJson);

                            var bkDesignIds = customBkDataService.getBkDesignIds(bkDataJson,
                                    block.getClass()).stream().sorted().collect(Collectors.toList());
                            var templateIds = block.getDesignTemplates().stream().map(DesignTemplates::getId)
                                    .sorted().collect(Collectors.toList());

                            var designs = bkData.path(customBkDataService.getDesignField(block.getClass()));

                            if (designs.isObject()) {
                                var castedDesign = (ObjectNode) designs;
                                for (int i = 0; i < bkDesignIds.size(); i++) {
                                    var prevIdKey = String.valueOf(bkDesignIds.get(i));
                                    if (!"0".equals(prevIdKey) && i < templateIds.size()) {
                                        var newId = templateIds.get(i);
                                        castedDesign.set(String.valueOf(newId),
                                                castedDesign.remove(prevIdKey));
                                    }
                                }
                            }

                            bkDataJson = objectMapper.writer(new DefaultPrettyPrinter())
                                    .writeValueAsString(bkData);

                            return new ModelChanges<>(block.getId(), BlockWithCustomBkDataAndDesignTemplates.class)
                                    .process(bkDataJson, BlockWithCustomBkDataAndDesignTemplates.BK_DATA)
                                    .applyTo(block);
                        } catch (JsonProcessingException e) {
                            throw new RuntimeException("Could not parse bk data", e);
                        }
                    })
                    .collect(Collectors.toList());

            blockModifyRepository.update(addContainer, appliedBkDataChanges);
        }
    }
}
