package ru.yandex.partner.core.entity.block.type.custombkdata;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.partner.core.entity.block.model.BlockWithCustomBkDataAndDesignTemplates;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;
import ru.yandex.partner.core.entity.designtemplates.model.DesignTemplates;

import static ru.yandex.partner.core.entity.block.model.prop.BlockWithCustomBkDataBkDataPropHolder.BK_DATA;
import static ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefects.badDesigns;

@Component
public class BlockWithCustomBkDataAndDesignTemplatesValidatorProvider {

    private final CustomBkDataService customBkDataService;

    public BlockWithCustomBkDataAndDesignTemplatesValidatorProvider(CustomBkDataService customBkDataService) {
        this.customBkDataService = customBkDataService;

    }

    public <M extends BlockWithCustomBkDataAndDesignTemplates> Validator<M, Defect> validator() {
        return block -> {
            var designTemplatesIds = StreamEx.of(Objects.requireNonNullElseGet(block.getDesignTemplates(),
                    Collections::emptyList)).map(DesignTemplates::getId).toSet();
            var vb = ModelItemValidationBuilder.of(block);
            if (Boolean.TRUE.equals(block.getIsCustomBkData())) {
                vb.item(BK_DATA)
                        .check(bk -> {
                            try {
                                var bkDesignIds = customBkDataService.getBkDesignIds(bk, block.getClass());

                                var badDesignIds = new ArrayList<String>();

                                for (String bkDesignId : bkDesignIds) {
                                    if (!bkDesignId.matches("^(-?)\\d+$")) {
                                        badDesignIds.add(bkDesignId);
                                    }
                                }

                                if (!badDesignIds.isEmpty()) {
                                    return badDesigns(String.join(", ", badDesignIds));
                                }

                                var diff = Sets.difference(
                                        customBkDataService.filterLongDesignIds(bkDesignIds),
                                        designTemplatesIds
                                );
                                if (!diff.isEmpty()) {
                                    return badDesigns(diff.stream().map(String::valueOf)
                                            .collect(Collectors.joining(", ")));
                                }
                            } catch (BkDataException e) {
                                return new Defect<>(BlockDefectIds.JsonDefectIds.INCORRECT_JSON);
                            }
                            return null;

                        }, When.notNull());
            }
            return vb.getResult();
        };
    }
}
