package ru.yandex.partner.core.entity.block.type.custombkdata

import NPartner.Page.TPartnerPage.TBlock
import com.fasterxml.jackson.core.JsonProcessingException
import org.json.JSONException
import org.json.JSONObject
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Component
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder
import ru.yandex.partner.core.CoreConstants
import ru.yandex.partner.core.action.exception.DefectInfoWithMsgId.defect
import ru.yandex.partner.core.block.BlockType
import ru.yandex.partner.core.block.BlockUniqueIdConverter
import ru.yandex.partner.core.entity.block.model.BlockWithCustomBkData
import ru.yandex.partner.core.entity.block.model.prop.BlockWithCustomBkDataBkDataPropHolder
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.INCORRECT_AD_TYPE_SET
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.INCORRECT_DSP_TYPE
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.MISSING_AD_TYPE_SET
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.MISSING_KEY
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.NONSENSE_AD_TYPE_SET
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.NON_PARSEABLE_BK_DATA
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.SIZES_INCORRECT_OR_NOT_FOUND
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.SIZES_MISSING
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.UNKNOWN_DSP
import ru.yandex.partner.core.entity.dsp.model.Dsp
import ru.yandex.partner.libs.bs.json.BkDataConverter

@Component
open class BlockWithCustomBkDataValidatorProvider(
    @Value("\${validation.bkdata.ignoreEmptySizesForInterstitial:true}")
    private val ignoreEmptySizesForInterstitial: Boolean,
    private val customBkDataService: CustomBkDataService
) {
    private val bkDataConverter = BkDataConverter()
    private val LOGGER = LoggerFactory.getLogger(BlockWithCustomBkDataValidatorProvider::class.java);

    fun <M : BlockWithCustomBkData?> validator(): Validator<M, Defect<*>> {
        return Validator { block: M ->
            val vb = ModelItemValidationBuilder.of(block)
            vb.item(BlockWithCustomBkDataBkDataPropHolder.BK_DATA)
                .check({
                    try {
                        JSONObject(it)
                        null
                    } catch (e: JSONException) {
                        Defect(BlockDefectIds.BkDataDefects.INCORRECT_CUSTOM_BK_DATA)
                    }
                }, When.notNull())

            vb.result
        }
    }

    fun <M : BlockWithCustomBkData> validateBlockBkData(
        allDsps: Map<Long, Dsp>,
        isValidatingBkData: (Long) -> Boolean,
        isCron: Boolean
    ) =
        Validator { block: M ->
            val vb = ModelItemValidationBuilder.of(block)

            if (isCron && isValidatingBkData(block.id).not()) {
                LOGGER.debug("Skip validating bk_data on block {}. Check BlockContainer.isValidatingBkData for more details", block.id)
                return@Validator vb.result
            }

            if (block.bkData == null) {
                LOGGER.debug("Skip validating bk_data on block {} because it's null", block.id)
                return@Validator vb.result
            }

            val bkDataVb = vb.item(BlockWithCustomBkDataBkDataPropHolder.BK_DATA)
            val blockBkDataParseResult = try {
                bkDataConverter.convertBlockJsonToProto(block.bkData)!!
            } catch (e: JsonProcessingException) {
                bkDataVb.check {
                    defect<Any>(NON_PARSEABLE_BK_DATA.format(e.message))
                }
                return@Validator vb.result
            }

            for ((path, error) in blockBkDataParseResult.errors) {
                path.fold(bkDataVb) { cvb, pathItem -> cvb.item(null, pathItem) }
                    .check { defect<Any>(NON_PARSEABLE_BK_DATA, error) }
            }

            if (bkDataVb.result.hasAnyErrors()) {
                return@Validator vb.result
            }

            val pageId = block.pageId
            val publicId = BlockUniqueIdConverter.publicId(block)

            val protoVb = ItemValidationBuilder.of<M, Defect<Any>>(block)

            protoVb.item(blockBkDataParseResult.message, "bk_data")
                .checkBy(bkDataValidator(pageId, publicId, allDsps))

            return@Validator protoVb.result
        }

    open fun bkDataValidator(pageId: Long, maybePublicId: String?, allDsps: Map<Long, Dsp>) =
        Validator<TBlock, Defect<Any>> { blockBkData ->
            val bkDataVb = ItemValidationBuilder.of<TBlock, Defect<Any>>(blockBkData)
            val publicId = maybePublicId ?: getPublicId(blockBkData, pageId)
            checkRtbHasDesigns(blockBkData, bkDataVb)
            checkDspTypes(publicId, blockBkData, bkDataVb, allDsps)
            checkMediaSize(pageId, publicId, blockBkData, bkDataVb)
            checkInterstitialAdTypeSets(pageId, publicId, blockBkData, bkDataVb)
            checkInterstitialSizes(pageId, publicId, blockBkData, bkDataVb)
            checkDirectLimit(pageId, publicId, blockBkData, bkDataVb)
            bkDataVb.result
        }

    private fun checkDirectLimit(
        pageId: Long,
        publicId: String,
        blockBkData: TBlock,
        bkDataVb: ItemValidationBuilder<TBlock, Defect<Any>>
    ) {
        // TODO многие пейджи сейчас не пройдут валидацию, если сделать поле обязательным,
        //   а null в свойстве не парсится и интерпретируется как "не указано"
        if (blockBkData.hasDirectLimit() && blockBkData.directLimit == null) {
            bkDataVb.item(null, "DirectLimit").check {
                defect(BkDataValidationMsg.DIRECT_LIMIT_INCORRECT, pageId, publicId)
            }
        }
    }

    fun getPublicId(blockBkData: TBlock, pageId: Long) =
        // TODO should know ALL model prefixes here
        "${blockBkData.blockModel}-$pageId-${blockBkData.blockID}"

    private fun checkDspTypes(
        publicId: String,
        blockBkData: TBlock,
        bkDataVb: ItemValidationBuilder<TBlock, Defect<Any>>,
        allDsps: Map<Long, Dsp>
    ) {
        if (!blockBkData.hasDSPType() || blockBkData.dspInfoList.isEmpty()) {
            return
        }

        bkDataVb.list(blockBkData.dspInfoList, "DSPInfo").apply {
            checkEach(isKnownDsp(allDsps.keys))
            checkEach(hasSupportedDspType(publicId, blockBkData.dspType, allDsps), When.isValid())
        }
    }

    private fun checkMediaSize(
        pageId: Long,
        publicId: String,
        blockBkData: TBlock,
        bkDataVb: ItemValidationBuilder<TBlock, Defect<Any>>
    ) {
        if (blockBkData.hasDSPType()
            && blockBkData.dspType == CoreConstants.DspTypes.DSP_MEDIA.bkId()
            && (!blockBkData.hasAdBlockBlock() || !blockBkData.adBlockBlock)
        ) {
            if (!blockBkData.hasWidth()) {
                bkDataVb.item(blockBkData.width, "Width")
                    .check { defect(SIZES_MISSING, pageId, publicId) }
            } else if (!blockBkData.hasHeight()) {
                bkDataVb.item(blockBkData.height, "Height")
                    .check { defect(SIZES_MISSING, pageId, publicId) }
            } else if (blockBkData.width == 0 && blockBkData.height == 0) {
                // empty size here
                bkDataVb.check {
                    defect(SIZES_MISSING, pageId, publicId)
                }
            }
        }
    }

    private fun checkInterstitialAdTypeSets(
        pageId: Long,
        publicId: String,
        blockBkData: TBlock,
        bkDataVb: ItemValidationBuilder<TBlock, Defect<Any>>
    ) {
        if (blockBkData.hasInterstitialBlock() && !blockBkData.interstitialBlock) {
            val creatives = setOf(
                TBlock.EAdType.TEXT,
                TBlock.EAdType.MEDIA,
                TBlock.EAdType.MEDIA_PERFORMANCE,
                TBlock.EAdType.VIDEO,
                TBlock.EAdType.VIDEO_NON_SKIPPABLE,
                TBlock.EAdType.VIDEO_PERFORMANCE,
                TBlock.EAdType.VIDEO_MOTION,
                TBlock.EAdType.AUDIO,
            )

            if (blockBkData.adTypeSetList.none { it.adType in creatives }) {
                bkDataVb.check {
                    defect(INCORRECT_AD_TYPE_SET, publicId, pageId, publicId)
                }
            }

            bkDataVb.list(blockBkData.adTypeSetList, "AdTypeSet")
                .checkEach(Constraint {
                    if (!it.hasAdType()) {
                        defect(INCORRECT_AD_TYPE_SET, it.adType, pageId, publicId)
                    } else null
                })
                .checkEach(Constraint {
                    if (!it.hasValue()) {
                        defect(NONSENSE_AD_TYPE_SET, it.adType, pageId, publicId)
                    } else null
                }, When.isValid())
        }

        if (blockBkData.hasInterstitialBlock() && blockBkData.interstitialBlock
            && blockBkData.adTypeSetList.isEmpty()
        ) {
            bkDataVb.list(blockBkData.adTypeSetList, "AdTypeSet").check {
                defect(MISSING_AD_TYPE_SET, publicId, pageId, publicId)
            }
        }
    }

    private fun isKnownDsp(allDspsIds: Set<Long>) =
        Constraint<TBlock.TDSPInfo, Defect<Any>> { dspInfo ->
            if (dspInfo.dspid !in allDspsIds) {
                defect(UNKNOWN_DSP, dspInfo.dspid.toString())
            } else null
        }

    private fun hasSupportedDspType(publicId: String, blockDspType: Int, allDsps: Map<Long, Dsp>) =
        Constraint<TBlock.TDSPInfo, Defect<Any>?> { dspInfo ->
            val bkTypes = allDsps[dspInfo.dspid]!!.types.map { 1.shl(it.toInt()) }
            var dspTypeMask = bkTypes.reduce(Int::or)

            var formatsBkTypes = emptyList<Int>();
            if (dspTypeMask.and(CoreConstants.DspTypes.DSP_MOBILE.bkId()) > 0) {
                formatsBkTypes = allDsps[dspInfo.dspid]!!.formats
                    .map { 1.shl(CoreConstants.DspFormats.TYPE_MAP[it]!!.toInt()) }

                val formatDspTypeMask = formatsBkTypes.reduce(Int::or)

                dspTypeMask = dspTypeMask.or(formatDspTypeMask)
            }

            if (dspTypeMask.and(blockDspType) == 0) {
                var text = "$dspTypeMask (types: ${bkTypes.joinToString(", ")}"
                text += if (formatsBkTypes.isNotEmpty()) {
                    " and formats: ${formatsBkTypes.joinToString(", ")})"
                } else {
                    ")"
                }
                defect(INCORRECT_DSP_TYPE, publicId, blockDspType, dspInfo.dspid, text)
            } else null
        }

    private val blocksWithDesignSettings = setOf(
        BlockType.CONTEXT_ON_SITE_DIRECT,
        BlockType.SEARCH_ON_SITE_DIRECT,
        BlockType.SEARCH_ON_SITE_PREMIUM,
        BlockType.SSP_RTB,
        BlockType.INTERNAL_CONTEXT_ON_SITE_DIRECT,
        BlockType.INTERNAL_SEARCH_ON_SITE_DIRECT,
        BlockType.INTERNAL_SEARCH_ON_SITE_PREMIUM,
        BlockType.MOBILE_APP_RTB,
    )

    private fun checkRtbHasDesigns(
        blockBkData: TBlock,
        bkDataVb: ItemValidationBuilder<TBlock, Defect<Any>>
    ) {
        val blockModelType = BlockType.from(blockBkData.blockModel)

        if (blockModelType !in blocksWithDesignSettings) {
            return
        }

        val designFieldName = customBkDataService.getDesignField(blockModelType) ?: return

        val designField = TBlock.getDescriptor().findFieldByName(designFieldName)

        if (!blockBkData.hasField(designField)) {
            bkDataVb.item(blockBkData.design, designFieldName)
                .check { defect(MISSING_KEY, designFieldName) }
        }
    }

    private fun checkInterstitialSizes(
        pageId: Long,
        publicId: String,
        blockBkData: TBlock,
        bkDataVb: ItemValidationBuilder<TBlock, Defect<Any>>
    ) {
        if (!ignoreEmptySizesForInterstitial && blockBkData.hasInterstitialBlock()) {
            bkDataVb.item(blockBkData.sizesList, "Sizes").check { sizes ->
                if (sizes.isEmpty()) {
                    defect(SIZES_INCORRECT_OR_NOT_FOUND, publicId, pageId, publicId)
                } else null
            }
        }
    }
}
