package ru.yandex.partner.core.entity.block.type.custombkdata;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.Nullable;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.block.BlockType;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.ContentBlock;
import ru.yandex.partner.core.entity.block.model.InternalMobileRtbBlock;
import ru.yandex.partner.core.entity.block.model.InternalRtbBlock;
import ru.yandex.partner.core.entity.block.model.MobileRtbBlock;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.service.entitymanager.EntityManager;

import static ru.yandex.partner.core.CoreConstants.BK_DEFAULT_DESIGN_NAME;
import static ru.yandex.partner.core.CoreConstants.BK_RTB_DESIGN_NAME;

@Service
public class CustomBkDataService {
    private static final Map<Class<? extends BaseBlock>, String> DESIGN_FIELDS =
            Map.of(
                    RtbBlock.class, BK_RTB_DESIGN_NAME,
                    InternalRtbBlock.class, BK_RTB_DESIGN_NAME,
                    MobileRtbBlock.class, BK_RTB_DESIGN_NAME,
                    InternalMobileRtbBlock.class, BK_RTB_DESIGN_NAME,
                    ContentBlock.class, BK_RTB_DESIGN_NAME
            );

    private static final Map<BlockType, String> DESIGN_FIELD_BY_BLOCK_TYPE =
            Map.of(
                    BlockType.CONTEXT_ON_SITE_RTB, BK_RTB_DESIGN_NAME,
                    BlockType.INTERNAL_CONTEXT_ON_SITE_RTB, BK_RTB_DESIGN_NAME,
                    BlockType.MOBILE_APP_RTB, BK_RTB_DESIGN_NAME,
                    BlockType.CONTEXT_ON_SITE_DIRECT, BK_DEFAULT_DESIGN_NAME,
                    BlockType.SEARCH_ON_SITE_DIRECT, BK_DEFAULT_DESIGN_NAME,
                    BlockType.SEARCH_ON_SITE_PREMIUM, BK_DEFAULT_DESIGN_NAME,
                    BlockType.INTERNAL_CONTEXT_ON_SITE_DIRECT, BK_DEFAULT_DESIGN_NAME,
                    BlockType.INTERNAL_SEARCH_ON_SITE_DIRECT, BK_DEFAULT_DESIGN_NAME,
                    BlockType.INTERNAL_SEARCH_ON_SITE_PREMIUM, BK_DEFAULT_DESIGN_NAME,
                    BlockType.CONTEXT_ON_SITE_CONTENT, BK_RTB_DESIGN_NAME
            );

    private final ObjectMapper objectMapper;

    public CustomBkDataService(
            ObjectMapper objectMapper,
            EntityManager entityManager
    ) {
        Set<Class<? extends BaseBlock>> missedBlocks = entityManager.getMissedBlocks(DESIGN_FIELDS.keySet());
        if (!missedBlocks.isEmpty()) {
            throw new IllegalStateException("%s's field %s has missing mapping of %s"
                    .formatted(this.getClass(), "DESIGN_FIELDS", missedBlocks));
        }
        this.objectMapper = objectMapper;
    }

    public JsonNode getDataFromBk(String json) {
        return parseSneaky(json);
    }

    public JsonNode getDataFromBk(String json, String path) {
        return parseSneaky(json).path(path);
    }

    public Set<String> getBkDesignIds(String bkDataJson, Class<? extends BaseBlock> blockClass) {
        var designField = getDesignField(blockClass);

        var designs = getDataFromBk(bkDataJson, designField);
        if (!designs.isMissingNode() && !designs.isNull() && designs.isObject()) {
            return StreamEx.of(designs.fieldNames()).toSet();
        }
        return Collections.emptySet();
    }

    public Set<Long> filterLongDesignIds(Set<String> bkDesignIds) {
        return bkDesignIds.stream()
                .map(Long::valueOf)
                .filter(id -> id != 0L)
                .collect(Collectors.toSet());
    }

    public String getDesignField(Class<? extends BaseBlock> blockClass) {
        return DESIGN_FIELDS.getOrDefault(blockClass, BK_DEFAULT_DESIGN_NAME);
    }

    @Nullable
    public String getDesignField(BlockType blockType) {
        if (blockType == null) {
            return null;
        }
        return DESIGN_FIELD_BY_BLOCK_TYPE.get(blockType);
    }

    private JsonNode parseSneaky(String bkDataJson) {
        try {
            return objectMapper.readTree(bkDataJson);
        } catch (JsonProcessingException e) {
            throw new BkDataException("Could not parse BK_DATA", e);
        }
    }

}
