package ru.yandex.partner.core.entity.block.type.custombkoptions;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithCustomBkOptions;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockSimpleDefectParam;
import ru.yandex.partner.core.entity.block.service.validation.type.AbstractBlockValidationTypeSupport;
import ru.yandex.partner.core.entity.custombkoptions.CustomBkOptionsTypedRepository;
import ru.yandex.partner.core.entity.custombkoptions.filter.CustomBkOptionsFilters;
import ru.yandex.partner.core.entity.custombkoptions.model.CustomBkOptions;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.BinaryOperator;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.custombkoptions.CustomBkOptionsStateFlag;

import static java.util.function.Predicate.not;

@Component
@ParametersAreNonnullByDefault
public class BlockWithCustomBkOptionsValidationTypeSupport
        extends AbstractBlockValidationTypeSupport<BlockWithCustomBkOptions> {

    private final CustomBkOptionsTypedRepository customBkOptionsTypedRepository;
    private final Set<ModelProperty<? extends Model, ?>> modelProperties = Set.of(CustomBkOptions.ID,
            CustomBkOptions.BK_NAME);
    private final CoreFilterNode<CustomBkOptions> notDeletedFilter;

    @Autowired
    public BlockWithCustomBkOptionsValidationTypeSupport(
            CustomBkOptionsTypedRepository customBkOptionsTypedRepository) {

        this.customBkOptionsTypedRepository = customBkOptionsTypedRepository;
        this.notDeletedFilter = CoreFilterNode.create(
                CustomBkOptionsFilters.MULTISTATE, FilterOperator.IN,
                not((Multistate<CustomBkOptionsStateFlag> multistate) ->
                        multistate.test(CustomBkOptionsStateFlag.DELETED))
        );
    }

    @Override
    public ValidationResult<List<BlockWithCustomBkOptions>, Defect> validate(
            BlockContainer container, ValidationResult<List<BlockWithCustomBkOptions>, Defect> vr) {

        return new ListValidationBuilder<>(vr)
                .checkEachBy(validator(container))
                .getResult();
    }

    private Validator<BlockWithCustomBkOptions, Defect> validator(BlockContainer container) {
        return block -> {
            var vb = ModelItemValidationBuilder.of(block);
            vb.item(BlockWithCustomBkOptions.CUSTOM_BK_OPTIONS).checkByFunction(
                    customBkOptionIds -> checkCustomBkOptions(container.getCustomBkOptions(), customBkOptionIds),
                    When.notNull());
            return vb.getResult();
        };
    }

    @Override
    public Class<BlockWithCustomBkOptions> getTypeClass() {
        return BlockWithCustomBkOptions.class;
    }

    @Override
    public void fillContainer(BlockContainer container, List<BlockWithCustomBkOptions> blocks) {
        fillCustomBkOptions(container, blocks);
    }

    @Override
    public void fillContainerFullDictionaries(BlockContainer container) {
        fillCustomBkOptions(container, notDeletedFilter);
    }

    private void fillCustomBkOptions(BlockContainer container, List<BlockWithCustomBkOptions> blocks) {
        var blockCustomBkOptions = blocks.stream()
                .map(BlockWithCustomBkOptions::getCustomBkOptions)
                .filter(Objects::nonNull)
                .flatMap(List::stream)
                .collect(Collectors.toSet());
        var filter = CoreFilterNode
                .create(CustomBkOptionsFilters.ID, FilterOperator.IN, blockCustomBkOptions)
                .addFilterNode(BinaryOperator.AND, notDeletedFilter);

        fillCustomBkOptions(container, filter);
    }

    private void fillCustomBkOptions(BlockContainer container, CoreFilterNode<CustomBkOptions> filter) {
        var customBkOptions = customBkOptionsTypedRepository.getAll(filter, modelProperties);
        container.setCustomBkOptions(customBkOptions.stream()
                .collect(Collectors.toMap(CustomBkOptions::getId, Function.identity())));
    }

    private Defect<BlockSimpleDefectParam> checkCustomBkOptions(
            Map<Long, CustomBkOptions> availableCustomBkOptions,
            List<Long> currentCustomBkOptionIds) {

        if (currentCustomBkOptionIds.isEmpty()) {
            return null;
        }

        // [ааa, ааабб] - это валидно
        // [ааа, ааа.ббб] - это НЕ валидно
        var names = new ArrayList<String>(currentCustomBkOptionIds.size());
        for (Long currentCustomBkOptionId : currentCustomBkOptionIds) {
            if (availableCustomBkOptions.containsKey(currentCustomBkOptionId)) {
                names.add(availableCustomBkOptions.get(currentCustomBkOptionId).getBkName() + ".");
            } else {
                return new Defect<>(BlockDefectIds.OptsDefectIds.SOME_OPTIONS_ARE_UNAVAILABLE);
            }
        }


        var sortedNames = names.stream()
                .sorted(Comparator.comparingInt(String::length))
                .toArray(String[]::new);

        for (int i = 0; i < sortedNames.length - 1; i++) {
            for (int j = i + 1; j < sortedNames.length; j++) {
                if (sortedNames[j].startsWith(sortedNames[i])) {
                    return new Defect<>(BlockDefectIds.OptsDefectIds.SOME_OPTIONS_HAVE_CONFLICT_CHANGES);
                }
            }
        }

        return null;
    }
}
