package ru.yandex.partner.core.entity.block.type.designtemplates;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import org.springframework.context.MessageSource;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithCommonShowVideo;
import ru.yandex.partner.core.entity.block.model.BlockWithDesignTemplates;
import ru.yandex.partner.core.entity.block.model.BlockWithDesignTemplatesAndCommonShowVideo;
import ru.yandex.partner.core.entity.block.repository.type.BlockRepositoryTypeSupportFacade;
import ru.yandex.partner.core.entity.block.service.type.update.AbstractBlockUpdateOperationTypeSupport;
import ru.yandex.partner.core.entity.designtemplates.model.DesignTemplates;
import ru.yandex.partner.core.messages.BlockWithDesignTemplatesMsg;
import ru.yandex.partner.dbschema.partner.enums.DesignTemplatesType;

import static java.util.function.Predicate.not;

@Order(100)
@Component
public class BlockWithDesignTemplatesAndCommonShowVideoUpdateOperationSupport
        extends AbstractBlockUpdateOperationTypeSupport<BlockWithDesignTemplatesAndCommonShowVideo> {
    private final MessageSourceAccessor messages;
    private final BlockContainerFiller blockContainerFiller;

    public BlockWithDesignTemplatesAndCommonShowVideoUpdateOperationSupport(
            MessageSource messageSource,
            BlockContainerFiller blockContainerFiller,
            BlockRepositoryTypeSupportFacade blockRepositoryTypeSupportFacade
    ) {
        super(blockRepositoryTypeSupportFacade);
        this.messages = new MessageSourceAccessor(messageSource);
        this.blockContainerFiller = blockContainerFiller;
    }

    @Override
    public void onChangesApplied(
            BlockContainer updateContainer,
            List<AppliedChanges<BlockWithDesignTemplatesAndCommonShowVideo>> appliedChanges) {
        blockContainerFiller.fillFeaturesAndOptions(
                updateContainer, Lists.transform(appliedChanges, AppliedChanges::getModel));

        for (AppliedChanges<BlockWithDesignTemplatesAndCommonShowVideo> changes : appliedChanges) {
            if (!changes.changed(BlockWithCommonShowVideo.SHOW_VIDEO)) {
                continue;
            }

            List<DesignTemplates> newDesignTemplates =
                    changes.getNewValue(BlockWithDesignTemplates.DESIGN_TEMPLATES);
            List<DesignTemplates> oldDesignTemplates =
                    changes.getOldValue(BlockWithDesignTemplates.DESIGN_TEMPLATES);
            var designTemplates = newDesignTemplates == null
                    ? oldDesignTemplates : newDesignTemplates;

            if (designTemplates == null) {
                continue;
            }

            if (!Boolean.TRUE.equals(changes.getNewValue(BlockWithCommonShowVideo.SHOW_VIDEO))) {
                changes.modify(
                        BlockWithDesignTemplates.DESIGN_TEMPLATES,
                        designTemplates.stream()
                                .filter(not(this::hasVideoType))
                                .collect(Collectors.toList())
                );
            } else if (designTemplates.stream().noneMatch(this::hasVideoType)) {
                var newValue = new ArrayList<>(designTemplates);
                newValue.add(getDefaultVideoDesign(changes.getModel()));
                changes.modify(
                        BlockWithDesignTemplates.DESIGN_TEMPLATES,
                        newValue
                );
            }
        }
    }

    private DesignTemplates getDefaultVideoDesign(BaseBlock block) {
        return new DesignTemplates()
                .withCaption(messages.getMessage(BlockWithDesignTemplatesMsg.DEFAULT_VIDEO_DESIGN))
                .withType(DesignTemplatesType.video)
                .withPageId(block.getPageId())
                .withBlockId(block.getBlockId())
                .withDesignSettings(Map.of("name", "inpage"));
    }

    private boolean hasVideoType(DesignTemplates dt) {
        return dt.getType() == DesignTemplatesType.video;
    }

    @Override
    public Class<BlockWithDesignTemplatesAndCommonShowVideo> getTypeClass() {
        return BlockWithDesignTemplatesAndCommonShowVideo.class;
    }
}
