package ru.yandex.partner.core.entity.block.type.designtemplates;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithDesignTemplates;
import ru.yandex.partner.core.entity.block.model.BlockWithDesignTemplatesAndDependsFields;
import ru.yandex.partner.core.entity.block.model.BlockWithSiteVersion;
import ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType;
import ru.yandex.partner.core.entity.block.type.designtemplates.designsettings.DesignSettingsAdjustService;
import ru.yandex.partner.core.entity.common.validation.ModelWithBlockPageIdValidatorProvider;
import ru.yandex.partner.core.entity.common.validation.NullPropertiesValidatorProvider;
import ru.yandex.partner.core.entity.designtemplates.model.DesignTemplates;
import ru.yandex.partner.core.service.msf.FormatSystemService;
import ru.yandex.partner.core.service.msf.MsfUtils;
import ru.yandex.partner.dbschema.partner.enums.DesignTemplatesType;

import static ru.yandex.partner.core.entity.block.model.prop.RtbBlockDesignTemplatesPropHolder.DESIGN_TEMPLATES;
import static ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefects.designTypeMustBySet;
import static ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefects.emptyDesignTemplates;
import static ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefects.invalidSetOfNativeDesign;
import static ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefects.tooManyDesigns;
import static ru.yandex.partner.core.feature.UserFeatureEnum.ALLOWED_DESIGN_AUCTION_NATIVE_ONLY;
import static ru.yandex.partner.core.feature.UserFeatureEnum.DESIGN_AUCTION_NATIVE;
import static ru.yandex.partner.core.feature.UserFeatureEnum.DESIGN_AUCTION_NATIVE_TURBO;
import static ru.yandex.partner.dbschema.partner.enums.DesignTemplatesType.media;
import static ru.yandex.partner.dbschema.partner.enums.DesignTemplatesType.native_;
import static ru.yandex.partner.dbschema.partner.enums.DesignTemplatesType.tga;
import static ru.yandex.partner.dbschema.partner.enums.DesignTemplatesType.video;

@ParametersAreNonnullByDefault
@Component
public class BlockWithDesignTemplatesAndDependsFieldsValidatorProvider {
    private static final Logger logger =
            LoggerFactory.getLogger(BlockWithDesignTemplatesAndDependsFieldsValidatorProvider.class);

    private static final Set<DesignTemplatesType> ALL_TYPES = Set.of(tga, native_, video, media);

    private static final Map<String, Set<DesignTemplatesType>> AVAILABLE_DESIGNS_BY_SITE_VERSION = Map.of(
            SiteVersionType.DESKTOP.getLiteral(), ALL_TYPES,
            SiteVersionType.MOBILE.getLiteral(), ALL_TYPES,
            SiteVersionType.TURBO.getLiteral(), ALL_TYPES,
            SiteVersionType.TURBO_DESKTOP.getLiteral(), ALL_TYPES,
            SiteVersionType.MOBILE_FULLSCREEN.getLiteral(), ALL_TYPES,
            SiteVersionType.MOBILE_REWARDED.getLiteral(), ALL_TYPES,
            SiteVersionType.MOBILE_FLOORAD.getLiteral(), ALL_TYPES,
            SiteVersionType.AMP.getLiteral(), Set.of(tga, media));

    private static final Map<String, String> NATIVE_DESIGN_BY_FEATURE = Map.of(
            SiteVersionType.DESKTOP.getLiteral(), DESIGN_AUCTION_NATIVE.getFeatureName(),
            SiteVersionType.MOBILE.getLiteral(), DESIGN_AUCTION_NATIVE.getFeatureName(),
            SiteVersionType.TURBO.getLiteral(), DESIGN_AUCTION_NATIVE_TURBO.getFeatureName(),
            SiteVersionType.TURBO_DESKTOP.getLiteral(), DESIGN_AUCTION_NATIVE_TURBO.getFeatureName()
    );

    private final DesignTemplatesValidatorProvider designTemplatesValidatorProvider;
    private final ModelWithBlockPageIdValidatorProvider<DesignTemplates> modelWithBlockPageIdValidatorProvider;
    private final NullPropertiesValidatorProvider<DesignTemplates> nullPropertiesValidatorProvider;
    private final DesignSettingsAdjustService designSettingsAdjustService;
    private final FormatSystemService formatSystemService;

    public BlockWithDesignTemplatesAndDependsFieldsValidatorProvider(
            DesignTemplatesValidatorProvider designTemplatesValidatorProvider,
            ModelWithBlockPageIdValidatorProvider<DesignTemplates> modelWithBlockPageIdValidatorProvider,
            NullPropertiesValidatorProvider<DesignTemplates> nullPropertiesValidatorProvider,
            DesignSettingsAdjustService designSettingsAdjustService, FormatSystemService formatSystemService) {
        this.designTemplatesValidatorProvider = designTemplatesValidatorProvider;
        this.modelWithBlockPageIdValidatorProvider = modelWithBlockPageIdValidatorProvider;
        this.nullPropertiesValidatorProvider = nullPropertiesValidatorProvider;
        this.designSettingsAdjustService = designSettingsAdjustService;
        this.formatSystemService = formatSystemService;
    }

    public static Set<DesignTemplatesType> getAvailableDesignTemplateTypes(String siteVersion,
                                                                           Set<String> featureAndOptions,
                                                                           boolean showVideo) {
        var typesBySite = AVAILABLE_DESIGNS_BY_SITE_VERSION.getOrDefault(siteVersion, Set.of());

        Set<DesignTemplatesType> types = new HashSet<>();
        types.add(tga);
        types.add(media);

        if (typesBySite.contains(video) && showVideo) {
            types.add(video);
        }
        if (typesBySite.contains(native_) && featureAndOptions.contains(
                NATIVE_DESIGN_BY_FEATURE.getOrDefault(siteVersion, null))) {
            types.add(native_);
        }

        return types;
    }

    public <M extends BlockWithDesignTemplatesAndDependsFields> Validator<M, Defect>
    validator(BlockContainer container) {

        return block -> {
            List<DesignTemplates> designTemplates = block.getDesignTemplates();
            var id = block.getId();
            var validationBuilder = ModelItemValidationBuilder.of(block);
            if (Objects.isNull(designTemplates) || designTemplates.isEmpty()) {
                validationBuilder.item(DESIGN_TEMPLATES).checkByFunction(ds -> emptyDesignTemplates());
                return validationBuilder.getResult();
            }
            var templatesByType = StreamEx.of(designTemplates).groupingBy(DesignTemplates::getType);

            var featuresAndOptions = container.getFeaturesAndOptions()
                    .getOrDefault(block.getPageId(), Set.of());

            var siteVersion = block.getSiteVersion();
            var isAdfox = block.getAdfoxBlock();

            if (templatesByType.containsKey(native_)) {

                if (Objects.nonNull(siteVersion)
                        && checkDesignTypeBySiteVersion(siteVersion) && checkFeature(siteVersion, featuresAndOptions)) {
                    if (!templatesByType.containsKey(tga)
                            && !featuresAndOptions.contains(ALLOWED_DESIGN_AUCTION_NATIVE_ONLY.getFeatureName())) {
                        validationBuilder.item(DESIGN_TEMPLATES).checkByFunction(ds -> designTypeMustBySet(tga));
                        return validationBuilder.getResult();
                    }
                } else {
                    validationBuilder.item(DESIGN_TEMPLATES).checkByFunction(ds -> invalidSetOfNativeDesign());
                    return validationBuilder.getResult();
                }
            }
            var showVideo = Boolean.TRUE.equals(block.getShowVideo());
            var defaultFormats = formatSystemService.getDefaultFormats(CoreConstants.Msf.RU_LANG,
                    MsfUtils.getMsfRole(container, block.getClass()),
                    siteVersion, isAdfox);
            Map<String, Integer> designMaxLimits = defaultFormats == null ? Map.of() : defaultFormats.getLimit();
            for (var entry : templatesByType.entrySet()) {
                var count = entry.getValue().size();
                var type = entry.getKey();
                DesignLimits limits = getDesignLimits(type, showVideo, featuresAndOptions, designMaxLimits);
                if (count < limits.min) {
                    validationBuilder.item(DESIGN_TEMPLATES).checkByFunction(ds -> designTypeMustBySet(type));
                    return validationBuilder.getResult();
                }
                if (count > limits.max) {
                    validationBuilder.item(DESIGN_TEMPLATES).checkByFunction(ds -> tooManyDesigns(type, limits.max));
                    return validationBuilder.getResult();
                }
            }

            if (validationBuilder.item(BlockWithDesignTemplatesAndDependsFields.SITE_VERSION)
                    .getResult().hasAnyErrors()
            ) {
                // site version is REQUIRED as valid for further formats validation
                return validationBuilder.getResult();
            }

            DesignTemplatesContainer dsContainer = new DesignTemplatesContainer()
                    .withSiteVersion(siteVersion)
                    .withValidationMode(container.getMode())
                    .withValidateAsManager(container.getCanValidateDesignAsManager(block.getClass()))
                    .withAvailableDesignTypes(
                            getAvailableDesignTemplateTypes(siteVersion,
                                    featuresAndOptions, showVideo));
            if (!container.getUnmodifiedTemplates().isEmpty()) {
                var curTemplates = container.getUnmodifiedTemplates().getOrDefault(id, List.of());
                dsContainer.setUnmodifiedTemplates(StreamEx.of(curTemplates)
                        .collect(Collectors.toMap(DesignTemplates::getId, it -> it)));
            }

            validationBuilder.list(DESIGN_TEMPLATES).checkEachBy(
                    designTemplatesValidatorProvider.validator(dsContainer)
            );
            return validationBuilder.getResult();
        };
    }

    public <M extends BlockWithDesignTemplates> Validator<M, Defect> addPreValidator() {
        return block -> {
            var vb = ModelItemValidationBuilder.of(block);
            vb.list(DESIGN_TEMPLATES)
                    .checkEachBy(nullPropertiesValidatorProvider.addPreValidator(List.of(DesignTemplates.ID)))
                    .checkEachBy(modelWithBlockPageIdValidatorProvider.addPreValidator());
            return vb.getResult();
        };
    }

    private boolean checkDesignTypeBySiteVersion(String siteVersion) {
        var availableDesigns = AVAILABLE_DESIGNS_BY_SITE_VERSION.getOrDefault(siteVersion, null);
        if (Objects.isNull(availableDesigns)) {
            logger.warn("There are not available design types by site version {}", siteVersion);
            return false;
        }
        return AVAILABLE_DESIGNS_BY_SITE_VERSION.get(siteVersion).contains(native_);
    }

    private boolean checkFeature(String siteVersion, Set<String> featuresAndOptions) {
        var mandatoryFeature = NATIVE_DESIGN_BY_FEATURE.getOrDefault(siteVersion, null);
        if (Objects.isNull(mandatoryFeature)) {
            return true;
        }

        return featuresAndOptions.contains(mandatoryFeature);
    }

    public Validator<ModelChanges<BlockWithDesignTemplatesAndDependsFields>, Defect>
    validatorBeforeApply(List<ModelChanges<BlockWithDesignTemplatesAndDependsFields>> container, Map<Long,
            BlockWithDesignTemplatesAndDependsFields> unmodifiedValidModels, BlockContainer blockContainer) {

        return mc -> {

            List<DesignTemplates> designTemplates = mc.getPropIfChanged(BlockWithDesignTemplates.DESIGN_TEMPLATES);
            var validationBuilder
                    = ModelChangesValidationBuilder.of(mc);

            if (designTemplates == null) {
                return validationBuilder.getResult();
            }

            var siteVersion = mc.getPropIfChanged(BlockWithSiteVersion.SITE_VERSION);
            if (siteVersion == null) {
                siteVersion = (unmodifiedValidModels.get(mc.getId())).getSiteVersion();
            }

            DesignTemplatesContainer dsContainer = new DesignTemplatesContainer()
                    .withSiteVersion(siteVersion)
                    .withValidationMode(blockContainer.getMode())
                    .withValidateAsManager(blockContainer.getCanValidateDesignAsManager(mc.getModelType()))
                    .withAvailableDesignTypes(AVAILABLE_DESIGNS_BY_SITE_VERSION.getOrDefault(siteVersion, Set.of()));

            validationBuilder.list(designTemplates, BlockWithDesignTemplates.DESIGN_TEMPLATES.name()).checkEachBy(
                    designTemplatesValidatorProvider.validateBeforeApply(dsContainer, designSettingsAdjustService)
            );
            return validationBuilder.getResult();
        };
    }


    private DesignLimits getDesignLimits(DesignTemplatesType type, boolean showVideo, Set<String> featuresAndOptions,
                                         Map<String, Integer> designMaxLimits) {
        var limits = new DesignLimits();
        var maxByMsf = designMaxLimits.get(type.getLiteral());
        if (type.equals(video)) {
            return showVideo ? limits.withLimits(1, Math.min(1, maxByMsf == null ? 1 : maxByMsf)) :
                    limits.withLimits(0, 0);
        }
        if (type.equals(media)) {
            return limits.withLimits(1, maxByMsf == null ? 1 : maxByMsf);
        }

        limits.setLimitsByDesignType(type,
                featuresAndOptions.contains(ALLOWED_DESIGN_AUCTION_NATIVE_ONLY.getFeatureName()));
        limits.max = maxByMsf != null ? maxByMsf : limits.max;
        return limits;

    }

    private static class DesignLimits {
        private Integer min;
        private Integer max;

        private DesignLimits withLimits(Integer min, Integer max) {
            this.min = min;
            this.max = max;
            return this;
        }

        private void setLimitsByDesignType(DesignTemplatesType type, boolean allowedDesignAuctionNativeOnly) {
            if (type.equals(tga)) {
                if (allowedDesignAuctionNativeOnly) {
                    this.min = 0;
                } else {
                    this.min = 1;
                }
                this.max = 20;
            } else if (type.equals(native_)) {
                this.min = 0;
                this.max = 20;
            } else {
                this.min = 0;
                this.max = 5;
            }
        }
    }
}
