package ru.yandex.partner.core.entity.block.type.designtemplates;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;

import NPartner.Page;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Maps;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.block.BlockUniqueIdConverter;
import ru.yandex.partner.core.entity.block.container.BlockBkDictContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithDesignTemplatesAndSiteVersion;
import ru.yandex.partner.core.entity.block.repository.BlockBkFiller;
import ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType;
import ru.yandex.partner.core.entity.designtemplates.model.DesignTemplates;
import ru.yandex.partner.dbschema.partner.enums.DesignTemplatesType;

import static ru.yandex.partner.libs.bs.ext.TBlockExt.getOrCreateAdTypeSet;

@Component
public class BlockWithDesignTemplatesAndSiteVersionBKFiller
        implements BlockBkFiller<BlockWithDesignTemplatesAndSiteVersion> {

    private final ObjectMapper objectMapper;
    private static final Set<String> EXCLUDED_SITE_VERSIONS = Set.of(SiteVersionType.MOBILE_REWARDED.getLiteral());

    @Autowired
    protected BlockWithDesignTemplatesAndSiteVersionBKFiller(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    @Override
    public void fillBkData(@NotNull BlockWithDesignTemplatesAndSiteVersion block,
                           @NotNull Page.TPartnerPage.TBlock.Builder bkData,
                           @NotNull BlockBkDictContainer container) {
        boolean hasVideo = false;
        boolean hasRewarded = false;
        boolean onlyRewarded = true;

        String publicId = BlockUniqueIdConverter.convertToPublicId(block.getId());
        List<DesignTemplates> designTemplates = Objects.requireNonNullElseGet(block.getDesignTemplates(), List::of);
        Map<Long, Map<String, Object>> bkDesign = Maps.newHashMapWithExpectedSize(designTemplates.size());
        for (DesignTemplates designTemplate : designTemplates) {
            String type = designTemplate.getType().getLiteral();
            String caption = designTemplate.getCaption();
            Map<String, Object> designSettings = new HashMap<>(designTemplate.getDesignSettings());
            designSettings.put("blockId", publicId);

            if (designTemplate.getType() == DesignTemplatesType.native_) {
                designSettings.put("name", "nativeDesign");

                int gridColumns = getAsInt(designSettings.remove("grid_columns"));
                int gridRows = getAsInt(designSettings.remove("grid_rows"));
                designSettings.put("limit", gridColumns * gridRows);

                String template = getAsString(designSettings.getOrDefault("template", ""))
                        .replace("${grid_columns}", "" + gridColumns)
                        .replace("${grid_rows}", "" + gridRows);
                designSettings.put("template", template);

                if (gridRows == 1 && template.contains("<ya-units-slider")) {
                    type = "native-carousel";
                }
            }

            if (designTemplate.getType() == CoreConstants.DEFAULT_DESIGN_TEMPLATES_TYPE) {
                if ("rewarded".equals(designSettings.get("name")) || isTrue(designSettings.get("rewarded"))) {
                    hasRewarded = true;
                } else {
                    onlyRewarded = false; // means only rewarded tga designs.
                }
            }
            hasVideo = hasVideo || designTemplate.getType() == DesignTemplatesType.video;


            Map<String, Object> designTemplateSettings = new HashMap<>();
            designTemplateSettings.put("name", caption);
            designTemplateSettings.put("type", type);
            designTemplateSettings.put("design", designSettings);

            if (designTemplate.getType() == DesignTemplatesType.native_) {
                designTemplateSettings.put("pcodekey", designTemplate.getPcodeKey());
                designTemplateSettings.put("nativeTemplateFactors", designTemplate.getNativeTemplateFactors());
            }
            bkDesign.put(designTemplate.getId(), designTemplateSettings);
        }

        String designJson = getDesignJson(bkDesign);
        // todo при появлении model, добавить в BlockType признак ртб или нет
        //  и на основе этого вызывать setRtbDesign или setDesign
        //  perl https://github.yandex-team.ru/partner/partner2/blob/46ae95ff0f8beaee4a4c63b54ee2182ecb2c94e9/
        //  lib/Application/Model/Role/Has/Block/DesignTemplates.pm#L184
        bkData.setRtbDesign(designJson);

        if (hasVideo) {
            bkData.setVideo(
                    Page.TPartnerPage.TBlock.TVideo.newBuilder()
                            .setCategoryID(0)
                            .setCountPositions(1)
                            .setType("inpage")
                            .build()
            );
            bkData.getPageImpOptionsBuilder()
                    .addDisable("allow-multiple-dsp-ads");
        }
        // floorad doesn't have video designs
        if (!EXCLUDED_SITE_VERSIONS.contains(block.getSiteVersion())) {
            if (hasVideo) {
                // overrides previous work in BlockWithRtbFieldsAndShowVideoBkFiller.kt:41
                getOrCreateAdTypeSet(bkData, Page.TPartnerPage.TBlock.EAdType.VIDEO_MOTION)
                        .setValue(false)
                        .build();
            }

            // only fullscreen can reach this
            if (hasRewarded) {
                getOrCreateAdTypeSet(bkData, Page.TPartnerPage.TBlock.EAdType.VIDEO)
                        .setValue(true);
                getOrCreateAdTypeSet(bkData, Page.TPartnerPage.TBlock.EAdType.VIDEO_PERFORMANCE)
                        .setValue(true);

                // TODO: might be wrong
                if (onlyRewarded) {
                    getOrCreateAdTypeSet(bkData, Page.TPartnerPage.TBlock.EAdType.TEXT)
                            .setValue(false);
                }
            }
        }

        // TODO удалить если согласимся в перле оставить один способ
        BlockWithDesignTemplatesBkDataHelper.fillAllowedImageTypes(bkData, designTemplates);
    }

    @Override
    public int order() {
        return Order.DESIGN_TEMPLATES_AND_SITE_VERSION.ordinal();
    }


    private boolean isTrue(@Nullable Object object) {
        if (object == null) {
            return false;
        }

        if (object == Boolean.TRUE) {
            return true;
        }

        if (object instanceof Number number) {
            return number.intValue() > 0;
        }

        throw new IllegalArgumentException("Unexpected type. Class: " + object.getClass() + ", Value: " + object);
    }

    private int getAsInt(Object object) {
        if (object instanceof Number number) {
            return number.intValue();
        }

        if (object instanceof String str) {
            // NumberFormatException if incorrect string
            return Integer.parseInt(str);
        }

        throw new IllegalArgumentException("Unexpected type. Class: " + object.getClass() + ", Value: " + object);
    }

    private String getAsString(Object object) {
        if (object instanceof String str) {
            return str;
        }

        throw new IllegalArgumentException("Unexpected type. Class: " + object.getClass() + ", Value: " + object);
    }

    private String getDesignJson(Map<Long, Map<String, Object>> bkDesign) {
        try {
            return objectMapper.writeValueAsString(bkDesign);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Error during serialize to json.", e);
        }
    }

    @Override
    public Class<BlockWithDesignTemplatesAndSiteVersion> getTypeClass() {
        return BlockWithDesignTemplatesAndSiteVersion.class;
    }


}

