package ru.yandex.partner.core.entity.block.type.designtemplates;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.Result;
import ru.yandex.partner.core.action.ActionPerformer;
import ru.yandex.partner.core.entity.PageBlockIds;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithDesignTemplates;
import ru.yandex.partner.core.entity.block.repository.type.AbstractBlockRepositoryTypeSupport;
import ru.yandex.partner.core.entity.block.repository.type.BlockRepositoryTypeSupportWithoutMapper;
import ru.yandex.partner.core.entity.designtemplates.actions.all.factories.DesignTemplatesActionAddFactory;
import ru.yandex.partner.core.entity.designtemplates.filter.DesignTemplatesFilters;
import ru.yandex.partner.core.entity.designtemplates.model.BaseDesignTemplates;
import ru.yandex.partner.core.entity.designtemplates.model.DesignTemplates;
import ru.yandex.partner.core.entity.designtemplates.service.DesignTemplatesService;
import ru.yandex.partner.core.entity.designtemplates.service.add.DesignTemplatesAddOperationFactory;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.holder.ModelPropertiesHolder;
import ru.yandex.partner.core.multistate.designtemplates.DesignTemplatesMultistate;
import ru.yandex.partner.core.utils.FunctionalUtils;
import ru.yandex.partner.core.utils.converter.MassResultConverter;

import static ru.yandex.partner.core.holder.ModelPropertiesHolder.fromModelProperties;

@Component
@ParametersAreNonnullByDefault
public class BlockWithDesignTemplatesRepositoryTypeSupport
        extends AbstractBlockRepositoryTypeSupport<BlockWithDesignTemplates>
        implements BlockRepositoryTypeSupportWithoutMapper<BlockWithDesignTemplates> {
    private final DesignTemplatesService designTemplatesService;
    private final DesignTemplatesAddOperationFactory addOperationFactory;
    private final DesignTemplatesActionAddFactory addFactory;
    private final ActionPerformer actionPerformer;
    private final Set<ModelProperty<? super BlockWithDesignTemplates, ?>> affectedModelProperties;
    private final Set<ModelProperty<? super BlockWithDesignTemplates, ?>> editableModelProperties;
    private final ObjectMapper objectMapper;

    @Autowired
    public BlockWithDesignTemplatesRepositoryTypeSupport(DSLContext dslContext,
                                                         DesignTemplatesService designTemplatesService,
                                                         DesignTemplatesAddOperationFactory addOperationFactory,
                                                         DesignTemplatesActionAddFactory addFactory,
                                                         ActionPerformer actionPerformer,
                                                         ObjectMapper objectMapper) {
        super(dslContext);
        this.designTemplatesService = designTemplatesService;
        this.addOperationFactory = addOperationFactory;
        this.addFactory = addFactory;
        this.actionPerformer = actionPerformer;
        this.affectedModelProperties = Set.of(BlockWithDesignTemplates.DESIGN_TEMPLATES);
        this.editableModelProperties = Set.of(BlockWithDesignTemplates.DESIGN_TEMPLATES);
        this.objectMapper = objectMapper;
    }

    @Override
    public Set<ModelProperty<? super BlockWithDesignTemplates, ?>> getAffectedModelProperties() {
        return affectedModelProperties;
    }

    @Override
    public Class<BlockWithDesignTemplates> getTypeClass() {
        return BlockWithDesignTemplates.class;
    }

    @Override
    public ModelPropertiesHolder
    getEditableModelProperties(BlockWithDesignTemplates model) {
        return fromModelProperties(new HashSet<>(editableModelProperties));
    }

    @Override
    public void insertToAdditionTables(DSLContext context, BlockContainer addModelParametersContainer,
                                       Collection<BlockWithDesignTemplates> models) {
        var newDesigns = StreamEx.of(models).toFlatList(BlockWithDesignTemplates::getDesignTemplates);
        var container = addOperationFactory
                .prepareAddOperationContainer(addModelParametersContainer.getIncomingFields());
        var result = addOperationFactory.createAddOperation(Applicability.FULL, newDesigns, container)
                .prepareAndApply();

        if (result.isSuccessful() && result.getErrorCount() == 0) {
            // TODO: move it PI-24925
            var data = MassResultConverter.<DesignTemplates>convertMassResult(result);
            var mappedDesigns = StreamEx.of(data.getResult()).toMap(it -> it.getResult().getId(), Result::getResult);
            var actionAdd = addFactory.createAction(mappedDesigns, Collections.emptyMap());
            actionPerformer.doActions(actionAdd);
        }
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BlockWithDesignTemplates> models) {
        var pageBlockIds = FunctionalUtils.mapList(models,
                block -> new PageBlockIds(block.getPageId(), block.getBlockId())
        );

        MetaFilter<BaseDesignTemplates, PageBlockIds> pageBlockIdFilter = DesignTemplatesFilters.PAGE_BLOCK_ID_FILTER;
        CoreFilterNode<BaseDesignTemplates> filter = pageBlockIdFilter.in(pageBlockIds)
                .and(DesignTemplatesFilters.MULTISTATE
                        .eq(new DesignTemplatesMultistate().toMultistateValue()));
        var blockDesignTemplatesByPageBlockIds = StreamEx.of(designTemplatesService
                .findAll(
                        QueryOpts.forClass(DesignTemplates.class)
                                .withFilter(filter))
        ).groupingBy(it -> new PageBlockIds(it.getPageId(), it.getBlockId()));

        for (BlockWithDesignTemplates block : models) {
            PageBlockIds key = new PageBlockIds(block.getPageId(), block.getBlockId());
            block.setDesignTemplates(blockDesignTemplatesByPageBlockIds.get(key));
        }
    }


}
