package ru.yandex.partner.core.entity.block.type.designtemplates;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.collect.MapDifference;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Streams;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.AbstractAddOperation;
import ru.yandex.direct.result.MassResult;
import ru.yandex.partner.core.action.ActionPerformer;
import ru.yandex.partner.core.action.TransitionAction;
import ru.yandex.partner.core.action.result.ActionsResult;
import ru.yandex.partner.core.block.BlockUniqueIdConverter;
import ru.yandex.partner.core.entity.IncomingFields;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.DesignTemplatesOperationMsg;
import ru.yandex.partner.core.entity.common.editablefields.EditableFieldsService;
import ru.yandex.partner.core.entity.designtemplates.actions.all.factories.DesignTemplatesActionAddFactory;
import ru.yandex.partner.core.entity.designtemplates.actions.all.factories.DesignTemplatesDeleteFactory;
import ru.yandex.partner.core.entity.designtemplates.actions.all.factories.DesignTemplatesEditFactory;
import ru.yandex.partner.core.entity.designtemplates.model.BaseDesignTemplates;
import ru.yandex.partner.core.entity.designtemplates.model.DesignTemplates;
import ru.yandex.partner.core.entity.designtemplates.service.add.DesignTemplatesAddOperationFactory;
import ru.yandex.partner.core.exceptions.ValidationException;
import ru.yandex.partner.core.multitype.repository.relation.RelationRepository;
import ru.yandex.partner.core.multitype.repository.relation.UpdateStrategy;
import ru.yandex.partner.core.utils.converter.MassResultConverter;

class DesignTemplatesUpdateStrategy implements UpdateStrategy<DesignTemplates> {
    private final ActionPerformer actionPerformer;
    private final DesignTemplatesDeleteFactory designTemplatesDeleteFactory;
    private final DesignTemplatesEditFactory designTemplatesEditFactory;
    private final DesignTemplatesActionAddFactory designTemplatesAddFactory;
    private final DesignTemplatesAddOperationFactory designTemplatesAddOperationFactory;
    private final EditableFieldsService<BaseDesignTemplates> editableFieldsService;

    DesignTemplatesUpdateStrategy(
            ActionPerformer actionPerformer,
            DesignTemplatesDeleteFactory designTemplatesDeleteFactory,
            DesignTemplatesEditFactory designTemplatesEditFactory,
            DesignTemplatesActionAddFactory designTemplatesAddFactory,
            DesignTemplatesAddOperationFactory designTemplatesAddOperationFactory,
            EditableFieldsService<BaseDesignTemplates> editableFieldsService) {
        this.actionPerformer = actionPerformer;
        this.designTemplatesDeleteFactory = designTemplatesDeleteFactory;
        this.designTemplatesEditFactory = designTemplatesEditFactory;
        this.designTemplatesAddFactory = designTemplatesAddFactory;
        this.designTemplatesAddOperationFactory = designTemplatesAddOperationFactory;
        this.editableFieldsService = editableFieldsService;
    }

    @Override
    public void performUpdate(
            RelationRepository<DesignTemplates> repository,
            Pair<MapDifference<Long, DesignTemplates>, List<DesignTemplates>> oldAndNewValues,
            IncomingFields incomingFields) {
        var diff = oldAndNewValues.getLeft();

        var removedIds = diff.entriesOnlyOnRight().values()
                .stream().map(DesignTemplates::getId).collect(Collectors.toSet());

        var affectedBlockIds = diff.entriesOnlyOnRight().values().stream()
                .map(this::extractBlockId)
                .collect(Collectors.toSet());
        var updatedEntries = streamTemplatesForUpdate(diff)
                .map(newEntityState -> generateModelChanges(
                        newEntityState,
                        incomingFields.getUpdatedFields(newEntityState)
                ))
                .collect(Collectors.toList());

        affectedBlockIds.addAll(streamTemplatesForUpdate(diff)
                .map(this::extractBlockId)
                .collect(Collectors.toSet())
        );

        var entriesToAdd = oldAndNewValues.getRight();

        affectedBlockIds.addAll(entriesToAdd.stream()
                .map(this::extractBlockId)
                .collect(Collectors.toSet())
        );

        var actions = new ArrayList<TransitionAction<DesignTemplates, ?, ?>>(2);

        if (!removedIds.isEmpty()) {
            actions.add(designTemplatesDeleteFactory.delete(removedIds));
        }
        if (!updatedEntries.isEmpty()) {
            actions.add(designTemplatesEditFactory.edit(updatedEntries));
        }

        if (!actions.isEmpty()) {
            var result = actionPerformer.doActions(
                    actions.toArray(new TransitionAction[0])
            );

            if (!result.isCommitted()) {
                throw createValidationException(affectedBlockIds, result);
            }
        }

        if (!entriesToAdd.isEmpty()) {
            var container = designTemplatesAddOperationFactory
                    .prepareAddOperationContainer(incomingFields);
            AbstractAddOperation<DesignTemplates, Long> addOperation = designTemplatesAddOperationFactory
                    .createAddOperation(Applicability.FULL, entriesToAdd, container);

            var addMassResult = addOperation.prepareAndApply();

            if (!addMassResult.isSuccessful()) {
                throw createValidationException(affectedBlockIds, addMassResult);
            }

            var entriesWithIds = entriesToAdd.stream()
                    .collect(Collectors.toMap(DesignTemplates::getId, Function.identity()));

            Map<Long, Set<ModelProperty<? super DesignTemplates, ?>>> modelChangesByIds = entriesToAdd.stream()
                    .collect(Collectors.toMap(
                            DesignTemplates::getId,
                            newEntity -> {
                                var entityFields = incomingFields.getUpdatedFields(newEntity);
                                return (entityFields == null ? DesignTemplates.allModelProperties() : entityFields)
                                        .stream()
                                        .map(prop -> (ModelProperty<? super DesignTemplates, ?>) prop)
                                        .collect(Collectors.toSet());
                            }
                    ));

            var actionAddResult = actionPerformer.doActions(
                    designTemplatesAddFactory.createAction(entriesWithIds, modelChangesByIds)
            );

            if (!actionAddResult.isCommitted()) {
                throw createValidationException(affectedBlockIds, actionAddResult);
            }
        }
    }

    private ValidationException createValidationException(Set<Long> affectedBlockIds, ActionsResult<?> result) {
        var affectedIds = Multimaps.<Class<? extends ModelWithId>, Long>newMultimap(
                new HashMap<>(),
                HashSet::new
        );
        affectedIds.putAll(RtbBlock.class, affectedBlockIds);
        return new ValidationException(
                DesignTemplatesOperationMsg.DESIGN_TEMPLATES_UPDATE_FAILED,
                affectedIds,
                result
        );
    }

    private ValidationException createValidationException(Set<Long> affectedBlockIds, MassResult<Long> result) {
        var asActionResult = MassResultConverter.convertToActionResult(DesignTemplates.class, result);

        return createValidationException(affectedBlockIds, asActionResult);
    }

    private Stream<DesignTemplates> streamTemplatesForUpdate(MapDifference<Long, DesignTemplates> diff) {
        return Streams.concat(
                diff.entriesDiffering()
                        .values().stream()
                        .map(MapDifference.ValueDifference::leftValue),
                diff.entriesOnlyOnLeft().values().stream()
        );
    }

    private long extractBlockId(DesignTemplates designTemplate) {
        return BlockUniqueIdConverter.convertToUniqueId(
                BlockUniqueIdConverter.Prefixes.CONTEXT_ON_SITE_RTB_PREFIX,
                designTemplate.getPageId(),
                designTemplate.getBlockId()
        );
    }

    /**
     * Собирает список изменения из сущности-патча и списка полей, которые в ней прислали.
     *
     * @param newEntityState сущность-патч
     * @param updatedFields  список изменённых полей, если null - считаем что обновляются все поля
     */
    private ModelChanges<DesignTemplates> generateModelChanges(
            DesignTemplates newEntityState,
            @Nullable Collection<ModelProperty<?, ?>> updatedFields
    ) {
        var mc = new ModelChanges<>(newEntityState.getId(), DesignTemplates.class);

        var editableProperties =
                editableFieldsService.calculateEditableModelPropertiesHolder(newEntityState, null);

        Collection<? extends ModelProperty<?, ?>> propsToEdit = (updatedFields == null
                ? DesignTemplates.allModelProperties() : updatedFields)
                .stream()
                .filter(editableProperties::isEnabled)
                .collect(Collectors.toSet());

        for (ModelProperty<?, ?> prop : propsToEdit) {
            ModelProperty<Model, Object> dtProp = (ModelProperty<Model, Object>) prop;
            mc.process(dtProp.get(newEntityState), dtProp);
        }

        return mc;
    }
}
