package ru.yandex.partner.core.entity.block.type.designtemplates;

import java.util.Map;
import java.util.Objects;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.partner.core.entity.block.service.OperationMode;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;
import ru.yandex.partner.core.entity.block.type.designtemplates.designsettings.DesignSettingsAdjustService;
import ru.yandex.partner.core.entity.block.type.designtemplates.designsettings.DesignSettingsBeforeApplyValidator;
import ru.yandex.partner.core.entity.block.type.designtemplates.designsettings.DesignSettingsValidatorFactory;
import ru.yandex.partner.core.entity.block.type.designtemplates.designsettings.DesignSettingsValidatorHelper;
import ru.yandex.partner.core.entity.designtemplates.model.DesignTemplates;
import ru.yandex.partner.core.service.msf.FormatSystemService;

import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isEqual;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.minStringLength;
import static ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefects.invalidDesignType;
import static ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefects.keyRequired;
import static ru.yandex.partner.core.entity.designtemplates.model.prop.CommonDesignTemplatesCaptionPropHolder.CAPTION;
import static ru.yandex.partner.core.entity.designtemplates.model.prop.CommonDesignTemplatesDesignSettingsPropHolder.DESIGN_SETTINGS;
import static ru.yandex.partner.core.entity.designtemplates.model.prop.CommonDesignTemplatesFilterTagsPropHolder.FILTER_TAGS;
import static ru.yandex.partner.core.entity.designtemplates.model.prop.CommonDesignTemplatesTypePropHolder.TYPE;

@Component
public class DesignTemplatesValidatorProvider {
    private final FormatSystemService formatSystemService;
    private final DesignSettingsValidatorHelper settingsValidatorHelper;

    @Autowired
    public DesignTemplatesValidatorProvider(FormatSystemService formatSystemService,
                                            DesignSettingsValidatorHelper helper) {
        this.formatSystemService = formatSystemService;
        this.settingsValidatorHelper = helper;
    }

    public Validator<DesignTemplates, Defect> validator(DesignTemplatesContainer container) {

        return designTemplate -> {
            var validationBuilder = ModelItemValidationBuilder.of(designTemplate);
            validationBuilder.item(CAPTION)
                    .check(notNull())
                    .check(minStringLength(1), When.notNull())
                    .check(maxStringLength(255), When.notNull());
            validationBuilder.item(TYPE)
                    .check(notNull())
                    .check(inSet(container.getAvailableDesignTypes()), invalidDesignType(designTemplate.getType()),
                            When.notNull());
            validationBuilder.list(FILTER_TAGS)
                    .checkEachBy(it ->
                            ItemValidationBuilder.of(it, Defect.class).check(minStringLength(1)).getResult());

            if (Objects.isNull(designTemplate.getType())
                    || !container.getAvailableDesignTypes().contains(designTemplate.getType())) {
                return validationBuilder.getResult();
            }

            var unmodifiedTemplate = container.getUnmodifiedTemplates();
            var unmodifiedValidModel = unmodifiedTemplate == null ?
                    null : unmodifiedTemplate.getOrDefault(designTemplate.getId(), null);
            Map<String, Object> unmodifiedSettings = Map.of();

            if (container.getMode().equals(OperationMode.EDIT) && Objects.nonNull(unmodifiedValidModel)) {
                validationBuilder.item(TYPE).check(isEqual(unmodifiedValidModel.getType(), new Defect<>(
                        BlockDefectIds.DesignTemplates.TYPE_FIELD_CHANGES)));
                unmodifiedSettings = unmodifiedValidModel.getDesignSettings();
            }

            var designSettings = designTemplate.getDesignSettings();
            if (Objects.isNull(designSettings) || designSettings.isEmpty()) {
                var res = validationBuilder.getResult();
                res.addError(keyRequired("design_settings"));
                return res;
            }

            var curType = designTemplate.getType();

            var designSettingsValidator = DesignSettingsValidatorFactory
                    .getValidator(curType, formatSystemService, settingsValidatorHelper,
                            container.getSiteVersion(), container.getMode(),
                            unmodifiedSettings, container.canValidateAsManager());

            validationBuilder.item(DESIGN_SETTINGS)
                    .checkBy(designSettingsValidator);

            return validationBuilder.getResult();
        };
    }

    public Validator<DesignTemplates, Defect> validateBeforeApply(
            DesignTemplatesContainer container, DesignSettingsAdjustService designSettingsAdjustService) {

        return designTemplate -> {
            var validationBuilder = ModelItemValidationBuilder.of(designTemplate);

            var designSettings = designTemplate.getDesignSettings();
            if (Objects.isNull(designSettings) || designSettings.isEmpty()) {
                return validationBuilder.getResult();
            }

            var designSettingsValidator = new DesignSettingsBeforeApplyValidator(formatSystemService,
                    container.getSiteVersion(), designSettingsAdjustService);

            validationBuilder.item(DESIGN_SETTINGS)
                    .checkBy(designSettingsValidator);

            return validationBuilder.getResult();
        };
    }

}
