package ru.yandex.partner.core.entity.block.type.designtemplates.designsettings;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.designtemplates.model.DesignTemplates;
import ru.yandex.partner.core.service.msf.FormatSystemService;
import ru.yandex.partner.core.service.msf.MsfAdjustService;
import ru.yandex.partner.core.service.msf.dto.FormatSettingDto;
import ru.yandex.partner.core.service.msf.dto.FormatWithSettingsDto;
import ru.yandex.partner.dbschema.partner.enums.DesignTemplatesType;

@Service
public class DesignSettingsAdjustService {
    private final FormatSystemService formatSystemService;
    private final MsfAdjustService msfAdjustService;

    public DesignSettingsAdjustService(FormatSystemService formatSystemService, MsfAdjustService msfAdjustService) {
        this.formatSystemService = formatSystemService;
        this.msfAdjustService = msfAdjustService;
    }
    public <B> void adjustTemplateSettings(Collection<B> blocks,
                                           Function<B, List<DesignTemplates>> designTemplatesExtractor,
                                           Function<B, String> siteVersionExtractor
    ) {
        Multimap<Pair<String, String>, DesignTemplates> designNameAndSiteVersionToDesignTemplates =
                getDesignNameAndSiteVersionToDesignTemplates(blocks, designTemplatesExtractor, siteVersionExtractor);

        for (Pair<String, String> key : designNameAndSiteVersionToDesignTemplates.keySet()) {
            var formatSettings = formatSystemService.getFormatSettings(
                    key.getLeft(),
                    CoreConstants.Msf.RU_LANG,
                    CoreConstants.Msf.MSF_MANAGER_ROLE,
                    key.getRight()
            );
            if (formatSettings == null) {
                continue;
            }
            var templatesToAdjustSettingsIn
                    = designNameAndSiteVersionToDesignTemplates.get(key);
            adjustTemplateSettings(formatSettings, templatesToAdjustSettingsIn);
        }
    }

    @NotNull
    private <B> Multimap<Pair<String, String>, DesignTemplates>
    getDesignNameAndSiteVersionToDesignTemplates(Collection<B> blocks,
                                                 Function<B, List<DesignTemplates>> designTemplatesExtractor,
                                                 Function<B, String> siteVersionExtractor) {
        Multimap<Pair<String, String>, DesignTemplates> designNameAndSiteVersionToDesignTemplates
                = Multimaps.newListMultimap(new HashMap<>(), ArrayList::new);

        for (var block : blocks) {

            var designTemplates = designTemplatesExtractor.apply(block);
            if (designTemplates == null) {
                continue;
            }

            var siteVersion = siteVersionExtractor.apply(block);

            for (DesignTemplates designTemplate : designTemplates) {
                if (designTemplate.getType() != DesignTemplatesType.tga) {
                    continue;
                }
                Object name = designTemplate.getDesignSettings().get("name");
                if (name == null) {
                    // без name не можем вытащить схему
                    continue;
                }

                designNameAndSiteVersionToDesignTemplates.put(
                        Pair.of(name.toString(), siteVersion),
                        designTemplate
                );
            }
        }
        return designNameAndSiteVersionToDesignTemplates;
    }
    @VisibleForTesting
    public void adjustTemplateSettings(FormatWithSettingsDto formatSettings,
            Collection<DesignTemplates> templatesToAdjustSettingsIn) {
        var allowedKeys = formatSettings.getSettings().stream()
                .map(FormatSettingDto::getName)
                .collect(Collectors.toSet());
        allowedKeys.add("name");
        for (var designTemplate : templatesToAdjustSettingsIn) {
            var designSettings = EntryStream.of(designTemplate.getDesignSettings())
//                    .filter(entry -> allowedKeys.contains(entry.getKey()))
                    .toMap();
            for (FormatSettingDto formatSetting : formatSettings.getSettings()) {
                designSettings.computeIfPresent(
                        formatSetting.getName(),
                        (k, v) -> msfAdjustService.adjustValue(formatSetting, v)
                );
            }
            designTemplate.setDesignSettings(designSettings);
        }
    }
    public void tryToAdjustValue(FormatSettingDto formatSetting, Object value) {
        msfAdjustService.adjustValue(formatSetting, value);
    }
}
