package ru.yandex.partner.core.entity.block.type.designtemplates.designsettings;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;
import ru.yandex.partner.core.entity.block.service.validation.defects.DesignSettingsDefectParams;
import ru.yandex.partner.core.service.msf.FormatSystemService;
import ru.yandex.partner.core.service.msf.dto.FormatSettingDto;


@ParametersAreNonnullByDefault
public class DesignSettingsBeforeApplyValidator implements Validator<Map<String, Object>, Defect> {
    private static final Map<String, Class<?>> JS_TO_JAVA_TYPE = Map.of(
            "boolean", Boolean.class,
            "integer", Integer.class,
            "number", Number.class,
            "string", String.class,
            "array", List.class,
            "object", Map.class
    );
    public static final String RU_LANG = "ru";
    public static final String MSF_MANAGER_ROLE = "manager";


    private final FormatSystemService formatSystemService;
    private final String siteVersion;
    private final DesignSettingsAdjustService designSettingsAdjustService;

    public DesignSettingsBeforeApplyValidator(FormatSystemService formatSystemService, String siteVersion,
                                              DesignSettingsAdjustService designSettingsAdjustService) {
        this.formatSystemService = formatSystemService;
        this.siteVersion = siteVersion;
        this.designSettingsAdjustService = designSettingsAdjustService;
    }

    @Override
    public ValidationResult<Map<String, Object>, Defect> apply(Map<String, Object> designSettings) {
        var validationBuilder
                = ItemValidationBuilder.of(designSettings, Defect.class);
        var name = designSettings.get("name");
        if (name == null) {
            return validationBuilder.getResult();
        }
        var formatSettings = formatSystemService.getFormatSettings(
                name.toString(),
                RU_LANG,
                MSF_MANAGER_ROLE,
                siteVersion
        );
        if (formatSettings == null) {
            return validationBuilder.getResult();
        }
        Map<String, FormatSettingDto> formats =
            formatSettings.getSettings().stream().filter(s -> designSettings.containsKey(s.getName()))
                .collect(Collectors.toMap(FormatSettingDto::getName, Function.identity()));
        for (String dsName: formats.keySet()) {
            validationBuilder.item(designSettings.get(dsName), dsName).checkByFunction(ds -> {
                FormatSettingDto formatSetting = formats.get(dsName);
                try {
                    designSettingsAdjustService.tryToAdjustValue(formatSetting, ds);
                } catch (Exception ex) {
                    return new Defect<>(BlockDefectIds.DesignSettings.DESIGN_SETTINGS_CANNOT_PARSE,
                            new DesignSettingsDefectParams().withKey(dsName)
                                    .withFormatJSType(formatSetting.getJsType()));
                }
                return null;
            });
        }
        return validationBuilder.getResult();
    }
}
