package ru.yandex.partner.core.entity.block.type.designtemplates.designsettings;

import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;
import ru.yandex.partner.core.entity.block.service.validation.defects.DesignTemplatesDefectParams;
import ru.yandex.partner.core.service.msf.FormatSystemService;
import ru.yandex.partner.core.service.msf.dto.FormatWithSettingsDto;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.partner.core.CoreConstants.DESIGN_SETTING_FLOORAD_PARAM;
import static ru.yandex.partner.core.CoreConstants.DESIGN_SETTING_FULLSCREEN_PARAM;
import static ru.yandex.partner.core.CoreConstants.DESIGN_SETTING_NAME_PARAM;
import static ru.yandex.partner.core.CoreConstants.DESIGN_SETTING_REWARDED_PARAM;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_FLOORAD;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_REWARDED;
import static ru.yandex.partner.core.validation.defects.TypeDefects.invalidTypeMustBeBoolean;
import static ru.yandex.partner.core.validation.defects.TypeDefects.invalidTypeMustBeHexadecimal;
import static ru.yandex.partner.core.validation.defects.TypeDefects.invalidTypeMustBeNumeric;


@ParametersAreNonnullByDefault
public class MediaDesignSettingsValidator implements Validator<Map<String, Object>, Defect> {
    private static final Pattern HEXADECIMAL_PATTERN = Pattern.compile("\\p{XDigit}+");
    private final String siteVersion;
    private final FormatSystemService formatSystemService;
    private final DesignSettingsValidatorHelper helper;

    public MediaDesignSettingsValidator(FormatSystemService formatSystemService, DesignSettingsValidatorHelper helper,
                                        String siteVersion) {
        this.formatSystemService = formatSystemService;
        this.helper = helper;
        this.siteVersion = siteVersion;
    }

    @Override
    public ValidationResult<Map<String, Object>, Defect> apply(Map<String, Object> designSettings) {
        var validationBuilder = ItemValidationBuilder.of(designSettings, Defect.class);
        validationBuilder.item(designSettings.getOrDefault("interscroller", null), "interscroller")
                .check(isBooleanType(), When.notNull());
        validationBuilder.item(designSettings.getOrDefault("filterSizes", null), "filterSizes")
                .check(isBooleanType(), When.notNull());
        validationBuilder.item(designSettings.getOrDefault("fullscreen", null), "fullscreen")
                .check(isBooleanType(), When.notNull());
        validationBuilder.item(designSettings.getOrDefault("horizontalAlign", null), "horizontalAlign")
                .check(isBooleanType(), When.notNull());
        validationBuilder.item(designSettings.getOrDefault("fullscreenDuration", null), "fullscreenDuration")
                .check(isNumericType(), When.notNull());
        validationBuilder.item(designSettings.getOrDefault("interscrollerBgColor", null), "interscrollerBgColor")
                .check(isHexadecimal(), When.notNull());

        FormatWithSettingsDto format = null;
        var designSettingName = (String) designSettings.get(DESIGN_SETTING_NAME_PARAM);
        if (designSettingName != null && !designSettingName.isBlank()) {
            format = formatSystemService.getFormatSettings(designSettingName,
                    CoreConstants.Msf.RU_LANG, CoreConstants.Msf.MSF_PARTNER_ROLE, siteVersion);
        }

        if (MOBILE_FLOORAD.getLiteral().equals(siteVersion)) {
            setDefaultFloorAd(designSettings, format);
            validationBuilder.item(designSettings.get(DESIGN_SETTING_FLOORAD_PARAM), DESIGN_SETTING_FLOORAD_PARAM)
                    .check(notNull())
                    .check(checkBooleanTrue());
        }


        if (MOBILE_REWARDED.getLiteral().equals(siteVersion)) {
            setDefaultRewarded(designSettings, format);
            setDefaultFullscreen(designSettings, format);
            validationBuilder.item(designSettings.get(DESIGN_SETTING_REWARDED_PARAM), DESIGN_SETTING_REWARDED_PARAM)
                    .check(notNull())
                    .check(checkBooleanTrue());
        }


        validationBuilder.checkByFunction(ds -> {
            var tempInterScroller = designSettings.getOrDefault("interscroller", null);
            boolean interscroller = false;
            if (tempInterScroller instanceof Boolean) {
                interscroller = (boolean) tempInterScroller;
            } else {
                return null;
            }

            if (designSettings.containsKey("interscrollerBgColor")
                    && !interscroller) {
                return new Defect<>(BlockDefectIds.DesignTemplates.INVALID_VIDEO_SETTINGS,
                        new DesignTemplatesDefectParams().withKey("interscrollerBgColor")
                                .withDesignType("interscroller"));
            }
            return null;
        })
                .checkByFunction(ds -> {
                    var tempfullscreen = designSettings.getOrDefault("fullscreen", null);
                    boolean fullscreen = false;
                    if (tempfullscreen instanceof Boolean) {
                        fullscreen = (boolean) tempfullscreen;
                    } else {
                        return null;
                    }

                    if (designSettings.containsKey("fullscreenDuration")
                            && !fullscreen) {
                        return new Defect<>(BlockDefectIds.DesignTemplates.INVALID_VIDEO_SETTINGS,
                                new DesignTemplatesDefectParams().withKey("fullscreenDuration")
                                        .withDesignType("fullscreen"));
                    }
                    return null;
                });

        return validationBuilder.getResult();
    }

    private void setDefaultFullscreen(Map<String, Object> designSettings, FormatWithSettingsDto format) {
        setDefault(format, designSettings, DESIGN_SETTING_FULLSCREEN_PARAM, true);
    }

    private void setDefaultFloorAd(Map<String, Object> designSettings, FormatWithSettingsDto format) {
        setDefault(format, designSettings, DESIGN_SETTING_FLOORAD_PARAM, true);
    }


    private void setDefaultRewarded(Map<String, Object> designSettings, FormatWithSettingsDto format) {
        setDefault(format, designSettings, DESIGN_SETTING_REWARDED_PARAM, true);
    }

    private void setDefault(FormatWithSettingsDto format, Map<String, Object> designSettings, String designSettingParam,
                            Object defaultValue) {
        if (format != null) {
            defaultValue =
                    Optional.ofNullable(helper.getDefaultFromFormat(designSettingParam, format)).orElse(defaultValue);
        }
        designSettings.putIfAbsent(designSettingParam, defaultValue);
    }


    private Constraint<Object, Defect> isBooleanType() {
        return it -> {
            if (!(it instanceof Boolean)) {
                return invalidTypeMustBeBoolean();
            }
            return null;
        };
    }

    private Constraint<Object, Defect> isNumericType() {
        return it -> {
            if (!(it instanceof Number)) {
                try {
                    Integer.valueOf(it.toString());
                } catch (NumberFormatException e) {
                    return invalidTypeMustBeNumeric();
                }
            }
            return null;
        };
    }

    private Constraint<Object, Defect> isHexadecimal() {
        return it -> {
            if (!(HEXADECIMAL_PATTERN.matcher(it.toString()).matches())) {
                return invalidTypeMustBeHexadecimal();
            }
            return null;
        };
    }

    public static Constraint<Object, Defect> checkBooleanTrue() {
        return it -> {
            if (!(it instanceof Boolean)) {
                return invalidTypeMustBeBoolean();
            } else if (!(Boolean) it) {
                return invalidValue();
            }
            return null;
        };
    }
}
