package ru.yandex.partner.core.entity.block.type.designtemplates.designsettings;

import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;

import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;

import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefects.keyRequired;
import static ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefects.unknownKey;
import static ru.yandex.partner.core.validation.defects.ids.TypeDefectIds.DATA_MUST_BE_INTEGER_NUMBER;


@ParametersAreNonnullByDefault
public class NativeDesignSettingsValidator implements Validator<Map<String, Object>, Defect> {
    private static final String CSS = "css";
    private static final String TEMPLATE = "template";
    private static final String GRID_ROWS = "grid_rows";
    private static final String GRID_COLUMNS = "grid_columns";

    @Override
    public ValidationResult<Map<String, Object>, Defect> apply(Map<String, Object> designSettings) {
        var validationBuilder = ItemValidationBuilder.of(designSettings, Defect.class);
        var fields = designSettings.keySet();

        var standardFields = Set.of(CSS, TEMPLATE, GRID_COLUMNS, GRID_ROWS);

        var missedFields = Sets.difference(standardFields, fields);
        if (!missedFields.isEmpty()) {
            StreamEx.of(missedFields).forEach(fieldName -> validationBuilder.item(fieldName, fieldName).check(isNull(),
                    keyRequired(fieldName)));
            return validationBuilder.getResult();
        }

        var unknownFields = Sets.difference(fields, standardFields);

        if (!unknownFields.isEmpty()) {
            var res = validationBuilder.getResult();
            res.addError(unknownKey(
                    StreamEx.of(unknownFields).collect(Collectors.joining(", "))));
            return res;
        }

        validationBuilder.checkByFunction(ds -> {
            var gridColumns = parseInt(ds.getOrDefault(GRID_COLUMNS, 0));
            var gridRows = parseInt(ds.getOrDefault(GRID_ROWS, 0));
            if (Objects.isNull(gridColumns) || Objects.isNull(gridRows)) {
                return new Defect<>(DATA_MUST_BE_INTEGER_NUMBER);
            }

            if (gridColumns * gridRows > 25) {
                return new Defect<>(BlockDefectIds.DesignTemplates.INVALID_GREED);
            }

            return null;
        });

        return validationBuilder.getResult();
    }

    private @Nullable Integer parseInt(Object data) {
        Integer result;
        if (data instanceof Integer) {
            result = (int) data;
        } else if (data instanceof String) {
            try {
                result = Integer.parseInt((String) data);
            } catch (Exception e) {
                e.printStackTrace();
                result = null;
            }
        } else {
            result = null;
        }
        return result;
    }

}
