package ru.yandex.partner.core.entity.block.type.designtemplates.designsettings;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.block.service.OperationMode;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;
import ru.yandex.partner.core.service.msf.FormatSystemService;
import ru.yandex.partner.core.service.msf.dto.FormatSettingType;
import ru.yandex.partner.core.service.msf.dto.MsfValidationResultDto;

import static ru.yandex.partner.core.CoreConstants.DESIGN_SETTING_FLOORAD_PARAM;
import static ru.yandex.partner.core.CoreConstants.DESIGN_SETTING_FULLSCREEN_PARAM;
import static ru.yandex.partner.core.CoreConstants.DESIGN_SETTING_LAYOUT_PARAM;
import static ru.yandex.partner.core.CoreConstants.DESIGN_SETTING_LIMIT_PARAM;
import static ru.yandex.partner.core.CoreConstants.DESIGN_SETTING_NAME_PARAM;
import static ru.yandex.partner.core.CoreConstants.DESIGN_SETTING_REWARDED_PARAM;
import static ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefects.msfDefect;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_FLOORAD;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_FULLSCREEN;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_REWARDED;


@ParametersAreNonnullByDefault
public class TgaDesignSettingsValidator implements Validator<Map<String, Object>, Defect> {


    private final DesignSettingsValidatorHelper helper;
    private final FormatSystemService formatSystemService;
    private final String siteVersion;
    private final OperationMode mode;
    private final Map<String, Object> unmodifiedSettings;
    private final boolean canValidateAsManager;

    @Autowired
    public TgaDesignSettingsValidator(FormatSystemService formatSystemService, DesignSettingsValidatorHelper helper,
                                      String siteVersion, OperationMode mode,
                                      Map<String, Object> unmodifiedSettings, boolean canValidateAsManager) {
        this.formatSystemService = formatSystemService;
        this.siteVersion = siteVersion;
        this.mode = mode;
        this.unmodifiedSettings = unmodifiedSettings;
        this.canValidateAsManager = canValidateAsManager;
        this.helper = helper;
    }

    @VisibleForTesting
    static Map<String, Object> diff(OperationMode mode, Map<String, Object> designSettings,
                                    Map<String, Object> unmodifiedSettings) {
        if ((!mode.equals(OperationMode.EDIT) ||
                !unmodifiedSettings.containsKey(DESIGN_SETTING_NAME_PARAM) ||
                !designSettings.containsKey(DESIGN_SETTING_NAME_PARAM) ||
                !unmodifiedSettings.get(DESIGN_SETTING_NAME_PARAM)
                        .equals(designSettings.get(DESIGN_SETTING_NAME_PARAM)))) {
            return new HashMap<>(designSettings);
        }

        var diff = Maps.difference(designSettings, unmodifiedSettings);
        Map<String, Object> settingsToValidate = new HashMap<>(diff.entriesOnlyOnLeft());
        settingsToValidate.put(DESIGN_SETTING_NAME_PARAM, designSettings.get(DESIGN_SETTING_NAME_PARAM));
        for (var entry : diff.entriesDiffering().entrySet()) {
            if (!Objects.equals(entry.getValue().leftValue(), entry.getValue().rightValue())) {
                settingsToValidate.put(entry.getKey(), entry.getValue().leftValue());
            }
        }
        return settingsToValidate;
    }

    @Override
    public ValidationResult<Map<String, Object>, Defect> apply(Map<String, Object> designSettings) {
        var validationBuilder = ItemValidationBuilder.of(designSettings, Defect.class);

        if (!this.canValidateAsManager && !mode.equals(OperationMode.DUPLICATE)) {
            // Измененные поля + имя дизайна
            var settingsToValidate = diff(mode, designSettings, unmodifiedSettings);
            if (!settingsToValidate.isEmpty()) {
                // Тут в settingsToValidate могут докинуться поля (Например, limit)
                var notAvailableFields = calculateNotAvailableFields(settingsToValidate,
                        designSettings, siteVersion);
                if (!notAvailableFields.isEmpty()) {
                    notAvailableFields.forEach(it -> validationBuilder.item(it, it)
                            .checkByFunction(field -> new Defect<>(
                                    BlockDefectIds.DesignTemplates.NOT_AVAILABLE_DESIGN_SETTING)));
                    return validationBuilder.getResult();
                }
            }
            //Валидируем дизайн через систему форматов под партнером
            var msfResult =
                    formatSystemService.validate(CoreConstants.Msf.RU_LANG, CoreConstants.Msf.MSF_PARTNER_ROLE,
                            siteVersion, settingsToValidate);
            if (!msfResult.isValid()) {
                processMsfResult(settingsToValidate, validationBuilder, msfResult);
            }
        }

        //Валидируем дизайн через систему форматов под менеджером
        var msfResult = formatSystemService.validate(CoreConstants.Msf.RU_LANG,
                CoreConstants.Msf.MSF_MANAGER_ROLE, siteVersion, designSettings);
        if (msfResult != null && !msfResult.isValid()) {
            processMsfResult(designSettings, validationBuilder, msfResult);
        }

        return validationBuilder.getResult();
    }

    @VisibleForTesting
    public List<String> calculateNotAvailableFields(Map<String, Object> settingsToValidate,
                                                     Map<String, Object> designSettings, String siteVersion) {
        var designSettingName = (String) designSettings.get(DESIGN_SETTING_NAME_PARAM);

        if (designSettingName == null || designSettingName.isBlank()) {
            // возвращаем что поле name не доступно если пустое
            return List.of(DESIGN_SETTING_NAME_PARAM);
        }

        var format = formatSystemService.getFormatSettings(designSettingName,
                CoreConstants.Msf.RU_LANG, CoreConstants.Msf.MSF_PARTNER_ROLE, siteVersion);
        if (format == null) {
            // ничего не можем сказать, такого формата не существует
            return List.of();
        }
        var availableFields = new HashSet<String>();
        var deps = new HashMap<String, Object>();
        availableFields.add(DESIGN_SETTING_NAME_PARAM);
        for (var settingDto : format.getSettings()) {
            availableFields.add(settingDto.getName());
            calculateDeps(deps, settingDto.getType(), settingDto.getName());
        }

        //Костыль, чтобы поправить баг https://st.yandex-team.ru/PI-29080, рабочее решение:
        //msf добавит ручку для проверки, можно ли пользователю менять те поля что он поменял, без проверки значений,
        //после чего уже будет полная валидация
        if (settingsToValidate.containsKey(DESIGN_SETTING_LAYOUT_PARAM)) {
            settingsToValidate.putIfAbsent(DESIGN_SETTING_LIMIT_PARAM, 2);
        }
        //Конец костыля

        settingsToValidate.putIfAbsent(DESIGN_SETTING_LIMIT_PARAM,
                    Optional.ofNullable(helper.getDefaultFromFormat(DESIGN_SETTING_LIMIT_PARAM, format)).orElse(1));

        if (MOBILE_FULLSCREEN.getLiteral().equals(siteVersion)) {
            settingsToValidate.putIfAbsent(DESIGN_SETTING_FULLSCREEN_PARAM,
                    Optional.ofNullable(helper.getDefaultFromFormat(DESIGN_SETTING_FULLSCREEN_PARAM, format))
                            .orElse(true));
        }

        if (MOBILE_FLOORAD.getLiteral().equals(siteVersion)) {
            settingsToValidate.putIfAbsent(DESIGN_SETTING_FLOORAD_PARAM,
                    Optional.ofNullable(helper.getDefaultFromFormat(DESIGN_SETTING_FLOORAD_PARAM, format))
                            .orElse(true));
        }

        if (MOBILE_REWARDED.getLiteral().equals(siteVersion)) {
            settingsToValidate.putIfAbsent(DESIGN_SETTING_REWARDED_PARAM,
                    Optional.ofNullable(helper.getDefaultFromFormat(DESIGN_SETTING_REWARDED_PARAM, format))
                            .orElse(true));
            settingsToValidate.putIfAbsent(DESIGN_SETTING_FULLSCREEN_PARAM,
                    Optional.ofNullable(helper.getDefaultFromFormat(DESIGN_SETTING_FULLSCREEN_PARAM, format))
                            .orElse(true));
        }

        var dependencies = settingsToValidate.keySet().stream()
                .map(deps::get)
                .filter(it -> Objects.nonNull(it) && it instanceof String)
                .collect(Collectors.toSet());
        dependencies.forEach(it -> settingsToValidate.put((String) it, designSettings.get(it)));

        return settingsToValidate.keySet().stream()
                .filter(o -> !availableFields.contains(o))
                .collect(Collectors.toList());
    }


    private void processMsfResult(Map<String, Object> designSettings,
                                  ItemValidationBuilder<Map<String, Object>, Defect> validationBuilder,
                                  MsfValidationResultDto msfResult) {
        if (msfResult.isValid()) {
            return;
        }
        msfResult.getMessages().forEach(it -> validationBuilder.checkByFunction(x -> msfDefect(it.getText())));
        for (var item : msfResult.getItems().entrySet()) {
            item.getValue().getMessages().forEach(it ->
                    validationBuilder.item(designSettings.get(item.getKey()), item.getKey())
                            .checkByFunction(x -> msfDefect(item.getKey(), it.getText()), When.isValid())
            );
        }
    }

    private void calculateDeps(HashMap<String, Object> deps, FormatSettingType type, String settingName) {
        var isEnabled = type.getIsEnabled();
        if (Objects.nonNull(isEnabled) && !isEnabled.isEmpty()) {
            var settings = isEnabled.get(1);
            if (settings instanceof List && ((List<?>) settings).get(0).equals("settings")) {
                deps.put(settingName, ((List<?>) settings).get(1));
            }
        }
    }



}
