package ru.yandex.partner.core.entity.block.type.dsps;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithDsps;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;
import ru.yandex.partner.core.entity.block.service.validation.defects.DspsDefectParams;
import ru.yandex.partner.core.entity.dsp.model.Dsp;
import ru.yandex.partner.core.entity.dsp.service.DspService;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.partner.core.validation.defects.TypeDefects.invalidTypeMustBeDefined;

@Component
public class BlockWithDspsValidatorProvider {
    private DspService dspService;

    @Autowired
    public BlockWithDspsValidatorProvider(DspService dspService) {
        this.dspService = dspService;
    }

    public <M extends BlockWithDsps> Validator<M, Defect> validator(BlockContainer container) {
        return block -> {
            var vb = ModelItemValidationBuilder.of(block);

            vb.item(BlockWithDsps.DSPS)
                    .check(notNull(), invalidTypeMustBeDefined())
                    .checkByFunction(l -> validateDspsList(container, block, l), When.notNull());

            if (block.getDsps() != null) {
                dspService.dspsFieldValidation(container, vb.getResult());

                vb.item(BlockWithDsps.DSPS)
                        .checkBy(new BlockDspsListValidator(), When.isValid());
            }

            return vb.getResult();
        };
    }

    private Defect validateDspsList(BlockContainer container, BlockWithDsps block, List<Dsp> dspList) {
        List<Long> blockDspsIds = StreamEx.of(dspList)
                .map(Dsp::getId)
                .toList();

        TreeSet<Long> unique = new TreeSet<>();
        Set<Long> duplicates = blockDspsIds.stream()
                .filter(d -> !unique.add(d))
                .collect(Collectors.toSet());

        if (!duplicates.isEmpty()) {
            return new Defect<>(BlockDefectIds.Dsps.HAS_DUPLICATES,
                    new DspsDefectParams().withDuplicates(duplicates)
            );
        }

        List<Dsp> availableDsps = container.getBlockAvailableDsps() != null ?
                container.getBlockAvailableDsps().get(block.getId()) : null;

        if (availableDsps != null && !availableDsps.isEmpty()) {
            Set<Long> availableDspsIds = StreamEx.of(availableDsps)
                    .map(Dsp::getId)
                    .toSet();

            if (availableDspsIds.isEmpty()) {
                throw new RuntimeException("Block without available dsps");
            }

            Sets.SetView<Long> difference = Sets.difference(new HashSet<>(blockDspsIds), availableDspsIds);
            if (!difference.isEmpty()) {
                return new Defect<>(BlockDefectIds.Dsps.HAS_NOT_FOUND,
                        new DspsDefectParams().withNotFound(difference));
            }
        } else {
            throw new RuntimeException("Block without available dsps");
        }

        return null;
    }
}
