package ru.yandex.partner.core.entity.block.type.dspsshowvideo;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import org.jooq.Condition;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithCommonShowVideo;
import ru.yandex.partner.core.entity.block.model.BlockWithDsps;
import ru.yandex.partner.core.entity.block.model.BlockWithDspsShowVideo;
import ru.yandex.partner.core.entity.block.model.BlockWithRtbFields;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;
import ru.yandex.partner.core.entity.dsp.model.Dsp;
import ru.yandex.partner.core.entity.dsp.rules.DspRule;
import ru.yandex.partner.core.entity.dsp.rules.DspRuleComponent;
import ru.yandex.partner.core.entity.dsp.rules.DspRuleContainer;
import ru.yandex.partner.core.entity.dsp.rules.DspRuleSet;
import ru.yandex.partner.core.entity.dsp.rules.result.DspRuleResult;
import ru.yandex.partner.core.entity.page.model.BasePage;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.core.CoreConstants.VIDEO_AWAPS_DSPS;
import static ru.yandex.partner.core.entity.block.model.prop.BlockWithDspsDspsPropHolder.DSPS;
import static ru.yandex.partner.dbschema.partner.Tables.CONTEXT_ON_SITE_RTB;
import static ru.yandex.partner.dbschema.partner.Tables.DSP;

public abstract class BlockWithDspsShowVideoDspRules<T extends BlockWithDspsShowVideo> extends DspRuleComponent<T> {

    public abstract boolean isShowVideoApplicable(T block, boolean canEditRichMedia);
    public abstract boolean getDefaultShowVideo(T block);

    public abstract Set<Long> getDefaultVideoDsps(T block);

    @Override
    public DspRuleSet getDspRuleSet(DspRuleContainer container) {
        return new DspRuleSet()
                .withRule(getShowVideoDspRule(container));
    }

    @Override
    public void fillDspRuleContainer(DspRuleContainer container, BaseBlock entity) {
        boolean showVideoApplicable = isShowVideoApplicable((T) entity, container.canEditRichMedia());
        boolean showVideo = !showVideoApplicable ? getDefaultShowVideo((T) entity) :
                Boolean.TRUE.equals(((BlockWithDspsShowVideo) entity).getShowVideo());
        if (showVideo) {
            container.addDspType(CoreConstants.DspTypes.DSP_VIDEO);
        }
    }

    @SuppressWarnings("unused")
    public DspRule<BasePage, T, Dsp> getShowVideoDspRule(DspRuleContainer container) {
        return new DspRule<>() {
            @Override
            public DspRuleResult defaultDsps(DspRuleContainer dspRuleContainer, BasePage page,
                                             T block) {
                boolean showVideoApplicable = isShowVideoApplicable(block, container.canEditRichMedia());
                boolean showVideo = !showVideoApplicable ? getDefaultShowVideo(block) :
                        Boolean.TRUE.equals(block.getShowVideo());
                if (showVideo) {
                    return DspRuleResult.extend(DSP.ID.in(dspRuleContainer.getVideoDspsForValidation(block)));
                } else {
                    return DspRuleResult.extendZero();
                }
            }

            @Override
            public DspRuleResult turnOnDsps(DspRuleContainer dspRuleContainer, BasePage page,
                                            T block, Dsp dsp) {
                Condition videoDspTurnOnBlockFilter = getVideoDspTurnOnBlockFilter(block, dsp);
                if (videoDspTurnOnBlockFilter != null) {
                    return DspRuleResult.limit(videoDspTurnOnBlockFilter);
                } else {
                    return DspRuleResult.limitIdentity();
                }
            }

            @Override
            @SuppressWarnings("rawtypes")
            public ValidationResult<T, Defect> dspsFieldValidation(
                    DspRuleContainer dspRuleContainer,
                    BlockContainer container,
                    ValidationResult<T, Defect> vr) {
                return new ItemValidationBuilder<>(vr)
                        .checkBy(block -> {
                            ModelItemValidationBuilder<T> vb =
                                    ModelItemValidationBuilder.of(block);
                            if (Objects.equals(block.getShowVideo(), Boolean.TRUE)) {
                                Set<Long> videoDspList = new HashSet<>(
                                        dspRuleContainer.getVideoDspsForValidation(block));
                                if (!videoDspList.isEmpty()) {
                                    vb.item(DSPS).checkByFunction(it -> {
                                        Set<Long> blockDsps = it.stream()
                                                .map(Dsp::getId)
                                                .collect(Collectors.toSet());
                                        if (Sets.intersection(blockDsps, videoDspList).isEmpty()) {
                                            return new Defect<>(BlockDefectIds.Dsps.EMPTY_DSPS_FOR_VIDEO_AD);
                                        }
                                        return null;
                                    });
                                }
                            }
                            return vb.getResult();
                        })
                        .getResult();
            }

            @Override
            public void dspsFieldEdit(DspRuleContainer dspRuleContainer,
                                      AppliedChanges<T> modelChanges) {
                List<Dsp> dsps = modelChanges.changed(DSPS) &&
                        modelChanges.getNewValue(DSPS) != null
                        ? modelChanges.getNewValue(DSPS)
                        : modelChanges.getOldValue(DSPS);

                if (dsps == null) {
                    return;
                }

                List<Long> dspsIds = dsps.stream().map(Dsp::getId).collect(Collectors.toList());

                Set<Long> videoDspIds = dspRuleContainer.getVideoDsps(modelChanges.getModel());

                List<Dsp> availableDsps = dspRuleContainer.getBlockAvailableDspsMap()
                        .get(modelChanges.getModel().getId());
                if (availableDsps == null || availableDsps.isEmpty()) {
                    throw new RuntimeException("Block available dsps is empty");
                }
                Set<Long> availableDspsIds = availableDsps.stream()
                        .map(Dsp::getId)
                        .collect(Collectors.toSet());

                if (!modelChanges.changed(BlockWithCommonShowVideo.SHOW_VIDEO)) {
                    return;
                }

                Objects.requireNonNull(dspRuleContainer.getDspMap(), "Need dsp cache in container!");

                if (!Objects.equals(modelChanges.getNewValue(BlockWithCommonShowVideo.SHOW_VIDEO), Boolean.TRUE)) {
                    modelChanges.modify(
                            BlockWithDsps.DSPS,
                            dsps.stream()
                                    .filter(d -> availableDspsIds.contains(d.getId()))
                                    .filter(not(d -> videoDspIds.contains(d.getId())))
                                    .collect(Collectors.toList())
                    );
                } else if (Objects.equals(modelChanges.getNewValue(BlockWithCommonShowVideo.SHOW_VIDEO), Boolean.TRUE)
                        && dspsIds.stream().noneMatch(videoDspIds::contains)
                ) {
                    var newDsps = new ArrayList<>(dsps);
                    newDsps.addAll(videoDspIds.stream()
                            .filter(availableDspsIds::contains)
                            .map(dspRuleContainer.getDspMap()::get)
                            // missing == deleted
                            .filter(Objects::nonNull)
                            .collect(Collectors.toList()));

                    modelChanges.modify(DSPS, newDsps);
                }
            }
        };
    }

    public DspRule<BasePage, T, Dsp> getSpecialShowVideoDspRule(DspRuleContainer container) {
        return new DspRule<>() {
            @Override
            public DspRuleResult availableDsps(DspRuleContainer dspRuleContainer, T block) {
                boolean showVideoApplicable = isShowVideoApplicable(block, container.canEditRichMedia());
                Boolean showVideo = !showVideoApplicable ? getDefaultShowVideo(block) :
                        Boolean.TRUE.equals(block.getShowVideo());
                if (Boolean.TRUE.equals(showVideo)) {
                    Set<Long> videoDsps = dspRuleContainer.getVideoDsps(block);
                    if (videoDsps.isEmpty()) {
                        return DspRuleResult.limitIdentity();
                    } else {
                        return DspRuleResult.extend(DSP.ID.in(videoDsps));
                    }
                } else {
                    return DspRuleResult.extendZero();
                }
            }
        };
    }

    private Condition getVideoDspTurnOnBlockFilter(T block, Dsp dsp) {
        if (block instanceof BlockWithRtbFields) {
            if (VIDEO_AWAPS_DSPS.contains(dsp.getId()) || getDefaultVideoDsps(block).contains(dsp.getId())) {
                return CONTEXT_ON_SITE_RTB.SHOW_VIDEO.eq(1L);
            }
        }
        return null;
    }
}
