package ru.yandex.partner.core.entity.block.type.geo;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import NPartner.Page.TPartnerPage;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.springframework.stereotype.Component;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapper.read.ReaderBuilders;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.block.container.BlockBkDictContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithGeo;
import ru.yandex.partner.core.entity.block.model.Geo;
import ru.yandex.partner.core.entity.block.repository.BlockBkFiller;
import ru.yandex.partner.core.entity.block.repository.type.AbstractBlockRepositoryTypeSupportWithMapper;
import ru.yandex.partner.core.entity.crimea.CrimeaRepository;
import ru.yandex.partner.core.entity.geo.GeoBaseDto;
import ru.yandex.partner.core.entity.geo.GeoBaseRepository;
import ru.yandex.partner.core.entity.geo.GeoService;
import ru.yandex.partner.core.holder.ModelPropertiesHolder;
import ru.yandex.partner.core.utils.CommonConverters;

import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.partner.core.holder.ModelPropertiesHolder.fromModelProperties;
import static ru.yandex.partner.dbschema.partner.Tables.CONTEXT_ON_SITE_RTB;

@Component
@ParametersAreNonnullByDefault
public class BlockWithGeoRepositoryTypeSupport
        extends AbstractBlockRepositoryTypeSupportWithMapper<BlockWithGeo>
        implements BlockBkFiller<BlockWithGeo> {

    private final GeoService geoService;
    private final GeoBaseRepository geoBaseRepository;
    private final CrimeaRepository crimeaRepository;
    private final JooqMapper<BlockWithGeo> jooqMapper;
    private final ModelPropertiesHolder editableModelProperties;


    public BlockWithGeoRepositoryTypeSupport(DSLContext dslContext, ObjectMapper objectMapper,
                                             GeoService geoService, GeoBaseRepository geoBaseRepository,
                                             CrimeaRepository crimeaRepository) {
        super(dslContext);
        this.geoService = geoService;
        this.geoBaseRepository = geoBaseRepository;
        this.crimeaRepository = crimeaRepository;
        this.jooqMapper = JooqMapperBuilder.<BlockWithGeo>builder()
                .map(convertibleProperty(BlockWithGeo.GEO, CONTEXT_ON_SITE_RTB.GEO,
                        value -> CommonConverters.jsonStringToList(objectMapper, value, Geo.class),
                        value -> CommonConverters.listToJsonString(objectMapper, value)))
                .readProperty(BlockWithGeo.HAS_GEO_SETTINGS,
                        ReaderBuilders.fromField(CONTEXT_ON_SITE_RTB.GEO).by(value -> {
                            // Заглушка
                            // Для классов PartnerRepositoryTypeSupportWithMapper метод getAffectedModelProperties
                            // служит (для возврощаемых полей):
                            //  - для проверки, что у всех полей есть Reader (JooqReader.getFieldsToRead)
                            //  - для определения, при запросе каких полей необходимо вызывать этот тайпсуппорт
                            return null;
                        }))
                .build();
        this.editableModelProperties = fromModelProperties(Set.of(BlockWithGeo.GEO));
    }

    @Override
    public JooqMapper<BlockWithGeo> getJooqMapper() {
        return jooqMapper;
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BlockWithGeo> models) {
        for (BlockWithGeo model : models) {
            model.setHasGeoSettings(model.getGeo() != null && model.getGeo().size() != 0);
        }
    }

    @Override
    public Class<BlockWithGeo> getTypeClass() {
        return BlockWithGeo.class;
    }

    @Override
    public ModelPropertiesHolder getEditableModelProperties(BlockWithGeo model) {
        return editableModelProperties;
    }


    @Override
    public void fillBkData(@NotNull BlockWithGeo model, @NotNull TPartnerPage.TBlock.Builder bkData,
                           @NotNull BlockBkDictContainer container) {
        if (model.getPageId() == null || model.getGeo() == null) {
            return;
        }

        boolean isCrimeaRussian = container.getPageIdsToIsCrimeaRussian().getOrDefault(model.getPageId(), false);

        List<Geo> geoList = geoService.getSubtreesWithRootsCpm(
                model.getGeo(), container.getGeoBase(isCrimeaRussian)
        );

        for (Geo geo : geoList) {
            long value = geo.getCpm().equals(CoreConstants.MAX_CPM) ?
                    CoreConstants.BLOCKED_CPM_VALUE.longValue() :
                    geo.getCpm().multiply(BigDecimal.valueOf(1000L)).longValue();
            bkData.addGeoBuilder()
                    .setGeoID(geo.getId())
                    .setCurrency(CoreConstants.Currency.DEFAULT)
                    .setValue(value);
        }
    }

    @Override
    public void fillContainer(@NotNull BlockBkDictContainer container, @NotNull List<? extends BlockWithGeo> models) {
        List<Long> pageIds = models.stream()
                .map(BaseBlock::getPageId)
                .filter(Objects::nonNull)
                .toList();

        container.withPageIdsToIsCrimeaRussian(crimeaRepository.getPageIdsToIsCrimeaRussian(pageIds));

        var rawGeoBase1 = geoBaseRepository.getRawGeoBase();
        var rawGeoBase2 = rawGeoBase1.stream()
                .map(it -> new GeoBaseDto(it.getId(), it.getParentId(), it.getRegularity(), it.getLevel()))
                .toList();
        container.withGeoBase(Map.of(
                false, geoService.getDtosWithFilledCrimea(false, rawGeoBase1),
                true, geoService.getDtosWithFilledCrimea(true, rawGeoBase2)
        ));
    }
}
