package ru.yandex.partner.core.entity.block.type.page;

import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import NPartner.Page.TPartnerPage;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.container.BlockBkDictContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithContextPage;
import ru.yandex.partner.core.entity.block.repository.BlockBkFiller;
import ru.yandex.partner.core.entity.block.repository.type.AbstractBlockRepositoryTypeSupport;
import ru.yandex.partner.core.entity.block.repository.type.BlockRepositoryTypeSupportWithoutMapper;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.ContextPage;
import ru.yandex.partner.core.entity.page.service.PageService;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.FilterOperator;

import static ru.yandex.partner.core.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class BlockWithContextPageRepositoryTypeSupport
        extends AbstractBlockRepositoryTypeSupport<BlockWithContextPage>
        implements BlockRepositoryTypeSupportWithoutMapper<BlockWithContextPage>,
        BlockBkFiller<BlockWithContextPage> {

    private final PageService pageService;
    private final Set<ModelProperty<? super BlockWithContextPage, ?>> affectedModelProperties;
    private final Set<ModelProperty<? extends Model, ?>> pageModelProperties;

    @Autowired
    public BlockWithContextPageRepositoryTypeSupport(DSLContext dslContext, PageService pageService) {
        super(dslContext);
        this.pageService = pageService;

        this.affectedModelProperties = Set.of(
                BlockWithContextPage.ID,
                BlockWithContextPage.CAMPAIGN
        );

        this.pageModelProperties = Set.of(
                ContextPage.ID,
                ContextPage.PAGE_ID,
                ContextPage.CAPTION,
                ContextPage.OWNER_ID,
                ContextPage.OWNER,
                ContextPage.DOMAIN_ID,
                ContextPage.DOMAIN,
                ContextPage.ALLOWED_AMP,
                ContextPage.ALLOWED_TURBO,
                ContextPage.MULTISTATE,
                ContextPage.UNMODERATED_RTB_AUCTION,
                ContextPage.ASSISTANTS,
                ContextPage.BLOCKS_COUNT,
                ContextPage.BLOCKS_LIMIT,
                ContextPage.IS_GRAYSITE
        );
    }

    @Override
    public Set<ModelProperty<? super BlockWithContextPage, ?>> getAffectedModelProperties() {
        return affectedModelProperties;
    }

    @Override
    public Class<BlockWithContextPage> getTypeClass() {
        return BlockWithContextPage.class;
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BlockWithContextPage> models) {
        Map<Long, ContextPage> contextPages = pageService.findAll(
                QueryOpts.forClass(ContextPage.class)
                        .withFilter(CoreFilterNode.create(
                                PageFilters.PAGE_ID,
                                FilterOperator.IN,
                                listToSet(models, BlockWithContextPage::getPageId)
                        ))
                        .withProps(pageModelProperties))
                .stream()
                .collect(Collectors.toMap(ContextPage::getId, Function.identity()));

        for (BlockWithContextPage model : models) {
            model.setCampaign(contextPages.get(model.getPageId()));
        }
    }

    @Override
    public void fillBkData(@NotNull BlockWithContextPage block,
                           @NotNull TPartnerPage.TBlock.Builder bkData,
                           @NotNull BlockBkDictContainer container) {
        String currency = Optional.ofNullable(block.getCampaign())
                .map(ContextPage::getOwner)
                .map(User::getCurrentCurrency)
                .orElse(CoreConstants.Currency.DEFAULT);

        bkData.setCPMCurrency(currency);
    }
}
