package ru.yandex.partner.core.entity.block.type.page;

import java.util.List;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithContextPage;
import ru.yandex.partner.core.entity.block.service.OperationMode;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockSimpleDefectParam;
import ru.yandex.partner.core.entity.block.service.validation.type.AbstractBlockValidationTypeSupport;

import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.partner.core.entity.page.service.validation.defects.PageDefectIds.ContextFieldDefectIds.LIMIT_EXCEEDED;

@Component
public class BlockWithContextPageValidationTypeSupport
        extends AbstractBlockValidationTypeSupport<BlockWithContextPage> {
    @Override
    public Class<BlockWithContextPage> getTypeClass() {
        return BlockWithContextPage.class;
    }

    @Override
    public ValidationResult<List<BlockWithContextPage>, Defect> addPreValidate(
            BlockContainer container,
            ValidationResult<List<BlockWithContextPage>, Defect> vr) {

        return new ListValidationBuilder<>(vr)
                .checkEachBy(addPreValidateBlocks(container),
                        When.valueIs(blockWithContextPage -> blockWithContextPage.getPageId() != null))
                .getResult();
    }

    @Override
    public ValidationResult<List<BlockWithContextPage>, Defect> validate(
            BlockContainer container,
            ValidationResult<List<BlockWithContextPage>, Defect> vr) {
        if (container.getMode() == OperationMode.ADD || container.getMode() == OperationMode.DUPLICATE) {
            return new ListValidationBuilder<>(vr)
                    .checkBy(addValidateBlocks())
                    .getResult();
        } else {
            return vr;
        }
    }

    private Validator<List<BlockWithContextPage>, Defect> addValidateBlocks() {
        return blocks -> {
            var vb = ListValidationBuilder.of(blocks, Defect.class);
            var pages = StreamEx.of(blocks).map(BaseBlock::getPageId).toSet();
            var curCount = StreamEx.of(pages).toMap(Function.identity(), it -> 1L);
            vb.checkEachBy(block -> {
                var builder = ItemValidationBuilder.of(block, Defect.class);
                var page = block.getCampaign();
                var added = curCount.get(block.getPageId());
                builder.item(page.getBlocksCount() + added, "BlocksCount")
                        .check(notGreaterThan(page.getBlocksLimit()), new Defect<>(LIMIT_EXCEEDED));
                curCount.put(block.getPageId(), ++added);
                return builder.getResult();
            });

            return vb.getResult();
        };

    }

    private Validator<BlockWithContextPage, Defect> addPreValidateBlocks(BlockContainer container) {
        return block -> {
            var vr = ModelItemValidationBuilder.of(block);

            vr.item(BlockWithContextPage.CAMPAIGN)
                    .checkByFunction(contextPage -> {
                        if (container.getReachablePages().get(block.getClass()).containsKey(block.getPageId())) {
                            return null;
                        } else {
                            return new Defect<>(
                                    BlockDefectIds.PageDefects.PAGE_NOT_FOUND,
                                    new BlockSimpleDefectParam().withParam(block.getPageId().toString())
                            );
                        }
                    });

            return vr.getResult();
        };
    }

}
