package ru.yandex.partner.core.entity.block.type.page;

import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import NPartner.Page.TPartnerPage;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.container.BlockBkDictContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithMobileAppSettings;
import ru.yandex.partner.core.entity.block.repository.BlockBkFiller;
import ru.yandex.partner.core.entity.block.repository.type.AbstractBlockRepositoryTypeSupport;
import ru.yandex.partner.core.entity.block.repository.type.BlockRepositoryTypeSupportWithoutMapper;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.MobileAppSettings;
import ru.yandex.partner.core.entity.page.service.PageService;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.FilterOperator;

import static ru.yandex.partner.core.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class BlockWithMobileAppSettingsRepositoryTypeSupport
        extends AbstractBlockRepositoryTypeSupport<BlockWithMobileAppSettings>
        implements BlockRepositoryTypeSupportWithoutMapper<BlockWithMobileAppSettings>,
        BlockBkFiller<BlockWithMobileAppSettings> {

    private final PageService pageService;
    private final Set<ModelProperty<? super BlockWithMobileAppSettings, ?>> affectedModelProperties;
    private final Set<ModelProperty<? extends Model, ?>> pageModelProperties;

    @Autowired
    public BlockWithMobileAppSettingsRepositoryTypeSupport(DSLContext dslContext, PageService pageService) {
        super(dslContext);
        this.pageService = pageService;

        this.affectedModelProperties = Set.of(
                BlockWithMobileAppSettings.ID,
                BlockWithMobileAppSettings.CAMPAIGN
        );

        this.pageModelProperties = Set.of(
                MobileAppSettings.ID,
                MobileAppSettings.MULTISTATE,
                MobileAppSettings.OWNER_ID,
                MobileAppSettings.OWNER,
                MobileAppSettings.PAGE_ID,
                MobileAppSettings.DOMAIN_ID,
                MobileAppSettings.DOMAIN,
                MobileAppSettings.UNMODERATED_RTB_AUCTION,
                MobileAppSettings.BLOCKS_COUNT,
                MobileAppSettings.BLOCKS_LIMIT,
                MobileAppSettings.TYPE,
                MobileAppSettings.STORE_ID,
                MobileAppSettings.ASSISTANTS
        );
    }

    @Override
    public Set<ModelProperty<? super BlockWithMobileAppSettings, ?>> getAffectedModelProperties() {
        return affectedModelProperties;
    }

    @Override
    public Class<BlockWithMobileAppSettings> getTypeClass() {
        return BlockWithMobileAppSettings.class;
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BlockWithMobileAppSettings> models) {
        Map<Long, MobileAppSettings> apps = pageService.findAll(
                QueryOpts.forClass(MobileAppSettings.class)
                        .withFilter(CoreFilterNode.create(
                                PageFilters.PAGE_ID,
                                FilterOperator.IN,
                                mapList(models, BlockWithMobileAppSettings::getPageId)
                        ))
                        .withProps(pageModelProperties))
                .stream()
                .collect(Collectors.toMap(MobileAppSettings::getId, Function.identity()));

        for (BlockWithMobileAppSettings model : models) {
            model.setCampaign(apps.get(model.getPageId()));
        }
    }

    @Override
    public void fillBkData(@NotNull BlockWithMobileAppSettings block,
                           @NotNull TPartnerPage.TBlock.Builder bkData,
                           @NotNull BlockBkDictContainer container) {
        String currency = Optional.ofNullable(block.getCampaign())
                .map(MobileAppSettings::getOwner)
                .map(User::getCurrentCurrency)
                .orElse(CoreConstants.Currency.DEFAULT);

        bkData.setCPMCurrency(currency);

    }
}
