package ru.yandex.partner.core.entity.block.type.pcode;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.service.msf.FormatSystemService;
import ru.yandex.partner.core.service.msf.MsfAdjustService;
import ru.yandex.partner.core.service.msf.dto.DefaultFormatDto;
import ru.yandex.partner.core.service.msf.dto.FormatSettingDto;

@Service
@ParametersAreNonnullByDefault
public class PCodeSettingsService {
    private static final Logger LOGGER = LoggerFactory.getLogger(PCodeSettingsService.class);
    private final FormatSystemService formatSystemService;
    private final MsfAdjustService msfAdjustService;

    @Autowired
    public PCodeSettingsService(FormatSystemService formatSystemService, MsfAdjustService msfAdjustService) {
        this.formatSystemService = formatSystemService;
        this.msfAdjustService = msfAdjustService;
    }

    public Map<String, Object> completeSettings(
            @Nullable Map<String, Object> pCodeSettings, @Nullable String siteVersion, String role, Boolean isAdfox) {

        if (siteVersion == null) {
            LOGGER.warn("SiteVersion is NULL");
            return pCodeSettings;
        }

        DefaultFormatDto defaultFormatDto = formatSystemService.getDefaultFormats(CoreConstants.Msf.RU_LANG,
                role, siteVersion, isAdfox);
        if (defaultFormatDto == null) {
            return pCodeSettings;
        }

        List<FormatSettingDto> pcodeSettingsDefaults = defaultFormatDto.getPcodeSettings();

        if (pcodeSettingsDefaults == null || pcodeSettingsDefaults.isEmpty()) {
            LOGGER.debug(
                    "Call for default formats did not return pcodeSettings field for siteVersion={}, role={}",
                    siteVersion, role
            );
            return pCodeSettings;
        }

        Map<String, Object> completePcodeSettings;
        if (pCodeSettings == null) {
            completePcodeSettings = Maps.newHashMapWithExpectedSize(pcodeSettingsDefaults.size());
        } else {
            completePcodeSettings = new HashMap<>(pCodeSettings);
        }

        for (FormatSettingDto pcodeSetting : pcodeSettingsDefaults) {
            var name = pcodeSetting.getName();
            if (completePcodeSettings.containsKey(name)) {
                var adjustValue = msfAdjustService.adjustValue(pcodeSetting, completePcodeSettings.get(name));
                completePcodeSettings.put(name, adjustValue);
            } else {
                var defaultValue = pcodeSetting.getType().getValue();
                if (defaultValue != null) {
                    completePcodeSettings.put(name, defaultValue);
                }
            }
        }

        return completePcodeSettings;

    }

}
