package ru.yandex.partner.core.entity.block.type.picategories;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import NPartner.Page.TPartnerPage;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.PageBlockIds;
import ru.yandex.partner.core.entity.block.container.BlockBkDictContainer;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithPiCategories;
import ru.yandex.partner.core.entity.block.model.InternalMobileRtbBlock;
import ru.yandex.partner.core.entity.block.model.InternalRtbBlock;
import ru.yandex.partner.core.entity.block.model.MobileRtbBlock;
import ru.yandex.partner.core.entity.block.model.PiCategory;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.block.repository.BlockBkFiller;
import ru.yandex.partner.core.entity.block.repository.type.AbstractBlockRepositoryTypeSupport;
import ru.yandex.partner.core.entity.block.repository.type.BlockRepositoryTypeSupportWithoutMapper;
import ru.yandex.partner.core.entity.picategories.PiCategoriesRepository;
import ru.yandex.partner.core.entity.picategories.filter.PiCategoryFilters;
import ru.yandex.partner.core.entity.utils.CpmUtils;
import ru.yandex.partner.core.holder.ModelPropertiesHolder;
import ru.yandex.partner.core.multitype.repository.relation.RelationRepository;
import ru.yandex.partner.core.utils.FunctionalUtils;

import static ru.yandex.partner.core.holder.ModelPropertiesHolder.fromModelProperties;

@Component
@ParametersAreNonnullByDefault
public class BlockWithPiCategoriesRepositoryTypeSupport
        extends AbstractBlockRepositoryTypeSupport<BlockWithPiCategories>
        implements BlockRepositoryTypeSupportWithoutMapper<BlockWithPiCategories>,
        BlockBkFiller<BlockWithPiCategories> {
    private static final Map<Class<? extends BaseBlock>, Set<String>> PRODUCT_TYPES = Map.of(
            RtbBlock.class, CoreConstants.RTB_PRODUCT_TYPES,
            InternalRtbBlock.class, CoreConstants.RTB_PRODUCT_TYPES,
            MobileRtbBlock.class, CoreConstants.RTB_PRODUCT_TYPES,
            InternalMobileRtbBlock.class, CoreConstants.RTB_PRODUCT_TYPES
    );
    private final PiCategoriesRepository piCategoriesRepository;
    private final Set<ModelProperty<? super BlockWithPiCategories, ?>> affectedModelProperties;
    private final Set<ModelProperty<? super BlockWithPiCategories, ?>> editableModelProperties;
    private final PiCategoriesDictService piCategoriesDictService;

    @Autowired
    public BlockWithPiCategoriesRepositoryTypeSupport(DSLContext dslContext,
                                                      PiCategoriesRepository piCategoriesRepository,
                                                      PiCategoriesDictService piCategoriesDictService) {
        super(dslContext);
        this.piCategoriesRepository = piCategoriesRepository;
        this.piCategoriesDictService = piCategoriesDictService;
        this.affectedModelProperties = Set.of(
                BlockWithPiCategories.HAS_PI_CATEGORIES_SETTINGS,
                BlockWithPiCategories.PI_CATEGORIES
        );
        this.editableModelProperties = Set.of(
                BlockWithPiCategories.PI_CATEGORIES
        );
    }

    @Override
    public Class<BlockWithPiCategories> getTypeClass() {
        return BlockWithPiCategories.class;
    }

    @Override
    public Set<ModelProperty<? super BlockWithPiCategories, ?>> getAffectedModelProperties() {
        return affectedModelProperties;
    }

    @Override
    public ModelPropertiesHolder getEditableModelProperties(BlockWithPiCategories model) {
        return fromModelProperties(new HashSet<>(editableModelProperties));
    }

    @Override
    public void insertToAdditionTables(DSLContext context, BlockContainer addModelParametersContainer,
                                       Collection<BlockWithPiCategories> models) {
        var additionalModels = StreamEx.of(models)
                .filter(model -> model.getPiCategories() != null)
                .toFlatList(BlockWithPiCategories::getPiCategories);
        piCategoriesRepository.insert(additionalModels);

    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BlockWithPiCategories> models) {
        Map<PageBlockIds, List<PiCategory>> result = RelationRepository.of(
                        piCategoriesRepository, PiCategoryFilters.pageId(), PiCategoryFilters.blockId())
                .select(dslContext,
                        FunctionalUtils.mapList(models, block ->
                                new PageBlockIds(block.getPageId(), block.getBlockId()))
                );

        for (BlockWithPiCategories block : models) {
            List<PiCategory> piCategories = result.get(new PageBlockIds(block.getPageId(), block.getBlockId()));
            block.setHasPiCategoriesSettings(piCategories != null && !piCategories.isEmpty());
            block.setPiCategories(piCategories != null ? piCategories : Collections.emptyList());
        }
    }

    @Override
    public void fillBkData(@NotNull BlockWithPiCategories block,
                           @NotNull TPartnerPage.TBlock.Builder bkData,
                           @NotNull BlockBkDictContainer container) {
        var productTypes = PRODUCT_TYPES.get(block.getClass());
        var iadData = CpmUtils.INSTANCE.getCpmFromTree(container.getPiCategoryDict(),
                block.getPiCategories().stream().collect(Collectors.toMap(PiCategory::getId, PiCategory::getCpm)));

        iadData.forEach(node -> {
            var builder = bkData.addPICategoryIABBuilder()
                    .setCategoryID(node.getId());
            var cpmValue = node.getValue();
            if (cpmValue == CoreConstants.MAX_CPM.intValue() * CoreConstants.CPM_MULTIPLIER) {
                cpmValue = CoreConstants.BLOCKED_CPM_VALUE.intValue();
            }

            if (productTypes.contains(CoreConstants.MEDIA_CREATIVE_REACH)) {
                builder.setMediaCreativeReach(cpmValue);
            }
            if (productTypes.contains(CoreConstants.MEDIA_IMAGE_REACH)) {
                builder.setMediaImageReach(cpmValue);
            }
            if (productTypes.contains(CoreConstants.VIDEO_CREATIVE_REACH)) {
                builder.setVideoCreativeReach(cpmValue);
            }
        });

    }

    @Override
    public void fillContainer(@NotNull BlockBkDictContainer container,
                              @NotNull List<? extends BlockWithPiCategories> models) {
        container.withPiCategoryDict(piCategoriesDictService.getCategoryDict());
    }
}
