package ru.yandex.partner.core.entity.block.type.rtbblock.external;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithContextPage;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.block.service.validation.type.AbstractBlockValidationTypeSupport;
import ru.yandex.partner.core.entity.block.type.base.PageReachabilityValidator;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.entity.page.model.ContextPage;
import ru.yandex.partner.core.entity.page.service.ReachablePageService;
import ru.yandex.partner.core.entity.page.service.RtbReachablePageService;
import ru.yandex.partner.core.filter.CoreFilterNode;

import static ru.yandex.partner.core.filter.CoreFilterNode.in;

@Component
public class RtbBlockValidationTypeSupport extends AbstractBlockValidationTypeSupport<BlockWithContextPage> {
    private final PageReachabilityValidator validatorProvider;
    private final ReachablePageService<RtbBlock> reachablePageService;

    @Autowired
    public RtbBlockValidationTypeSupport(
            PageReachabilityValidator validatorProvider,
            RtbReachablePageService reachablePageService) {
        this.validatorProvider = validatorProvider;
        this.reachablePageService = reachablePageService;
    }

    @Override
    public void fillContainer(BlockContainer container, List<BlockWithContextPage> models) {
        if (models.isEmpty()) {
            return;
        }

        // todo надо искать по реальному классу
        var blockClass = models.get(0).getClass();
        if (container.getReachablePages().containsKey(blockClass) && getTypeClass().isAssignableFrom(blockClass)) {
            container.getReachablePages().put(getTypeClass(), container.getReachablePages().get(blockClass));
        } else {
            container.getReachablePages().computeIfAbsent(getTypeClass(), clazz -> {
                Set<Long> pageIds = models.stream().map(BaseBlock::getPageId)
                        .filter(Objects::nonNull)
                        .collect(Collectors.toSet());

                if (pageIds.isEmpty()) {
                    return Collections.emptyMap();
                }

                CoreFilterNode<BasePage> pageFilter =
                        in(PageFilters.PAGE_ID, pageIds);

                if (container.getPageReachabilityFilter() != null) {
                    pageFilter = pageFilter.and((CoreFilterNode<BasePage>)
                            container.getPageReachabilityFilter(blockClass));
                }

                return reachablePageService.getReachablePages(
                        pageFilter,
                        ContextPage.class
                ).stream().collect(Collectors.toMap(BasePage::getId, Function.identity()));
            });
        }
    }

    @Override
    public Class<BlockWithContextPage> getTypeClass() {
        return BlockWithContextPage.class;
    }

    @Override
    public ValidationResult<List<BlockWithContextPage>, Defect> validate(
            BlockContainer container,
            ValidationResult<List<BlockWithContextPage>, Defect> vr) {

        var validator = validatorProvider.<BlockWithContextPage>validator(
                Objects.requireNonNull(container.getReachablePages().get(getTypeClass())));

        return new ListValidationBuilder<>(vr)
                .checkEachBy(validator)
                .getResult();
    }
}
